c======================= TTSOLVE25D_INV ============================
c This subroutine calculates all factors needed to do the
c Beylkin type inversion of reflection seismic data except
c the cosine of the angle between the incident and reflected
c waves at the reflection point.
c

      subroutine ttsolve25d_inv(nx,nz,dx,dz,
     &     xmin,zmin,xs,zs,zd,
     &     ap,v,tt,amps,tau_x,tau_z,phi_xr,tau_yy,
     &     work,len_work,ipdmp,idbg,ier)

c downsweep driver for 2D Cartesian grid traveltime calculations.
c initializes traveltime and amplitude at depth zd > zs, as point
c source traveltime and amplitude within ap degrees from
c vertical, and plane wave traveltime and amplitude for angles
c greater than arccos(cos(ap)/sqrt(2)) degrees.
c returns first arrival traveltime and geometric optics amplitudes.
c Next invokes TTAMP25D_INV which propagates tt tau_x, tau_z, tau_yy
c and amp downward with aperture-limited versions of eikonal and
c transport equations. Finally invokes:
c 1. BELK_DET which propagates the mixed (x, x_r) derivative of the traveltime
c by solving the convection equation.
c 2. SIGMA_GLOBAL which propagates the tau_yy derivative of the traveltime
c by solving the convection equation.
c 3. SIGMA_GLOBAL which propagates the out_of_plane component of the amplitude
c by solving the convection equation.

c Original (omnidirectional sweep) WWS 5.93
c
c Revised RV 2.94 added special initial conditions + velocity set +
c single downward traveltime sweep
c
c Rewritten WWS 7.94 
c Modified from the original program TTSOLVE Kidane Araya 9/1995.
c 
c WWS 20.10.95: now returns RECIPROCAL AMPLITUDES MULTIPLIED BY
c APERTURE CUTOFF - added routines to smooth aperture cutoff and
c reciprocate/scale
c
c WWS 24.10.95: now returns UNIT VECTORS for tt gradient components
c (tau_x, tau_z divided by vel)
c
c WWS 26.10.95: remove apcut from argument list
c
c WWS 30.10.95: now return in tau_z, tau_x the cos and sin resp.
c of ONE HALF the angle with the vertical ("psi")
c
c KARAYA 02.12.96: now includes 2.5D amplitude, tau_yy.
c
c Arguments:

      integer
     &     nx,nz,            ! number of samples in x and z
     &     len_work,         ! length of available workspace
     &     ipdmp,            ! dump unit number
     &     ier               ! error flag

      real 
     &     dx,dz,            ! steps
     &     xmin,zmin,        ! upper left corner coordinates
     &     xs,zs,            ! source coordinates
     &     zd,               ! datum depth
     &     ap,               ! faithful aperture
     &     v(*),             ! input velocity array
     &     tt(*),            ! output traveltimes
c NOTE: (WWS, 20.10.95)
c the amplitude field is now scaled by the aperture
     &     amps(*),          ! output GO amplitude
c NOTE: (WWS, 24.10.95):
c the following now scaled by slowness - so are (up to discretization
c error) UNIT VECTORS
     &     tau_x(*),         ! output GO x_component of gradient tt
     &     tau_z(*),         ! output GO z_component of gradient tt
     &     phi_xr(*),        ! output GO mixed x, x_r derivative of tt
     &     tau_yy(*),        ! output GO yy_component of Laplacian tt
c     &     apcut(*),         ! output GO aperture field
     &     work(*)           ! workspace

c internal variables

      integer
     &     idbg,             ! debug flag
     &     i,j,              ! loop counters
     &     size              ! size of global grid

      integer
     &     nsz,              ! depth index of datum
     &     new_work          ! remaining workspace

      integer
c WWS 26.10.95 - need pointer into work array for apcut
     &     apcut_ptr,        ! pointer to aperture cutoff array
     &     ssptr,            ! pointer to squared slowness array
     &     work_ptr          ! pointer to remaining workspace

      real 
     &     vmin,             ! minimum velocity
     &     tol,              ! zerodivide tolerance
     &     eik_eps,          ! eikonal threshhold
     &     sincut,           ! 
     &     tancut            ! 

c----------------------------------------------------------------------

      if (ier.ne.0) return

c set various parameters

      tol = 1.e-8

c size of global arrays

      size = nx*nz

c grab workspace for tmp arrays

      work_ptr = 1
      if (idbg.ge.1) then
         write(ipdmp,*)' TTSOLVE25D_INV ---> Get workspace...'
      end if
      call getbuf('work',ssptr,size,work_ptr,len_work,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: TTSOLVE25D_INV from GETBUF for ssptr'
         return
      end if
      call getbuf('work',apcut_ptr,size,work_ptr,len_work,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: TTSOLVE25D_INV from GETBUF for ssptr'
         return
      end if
      new_work=len_work

c BIG DEBUG DUMP

      if (idbg.ge.1) then
         write(ipdmp,1010)nx,nz,dx,dz,xmin,zmin,
     &        xs,zs,zd,ap
         write(ipdmp,1020)size,ssptr,apcut_ptr,work_ptr,len_work
      end if

 1010 format(
     & ' TTSOLVE25D_INV: debug dump',/,
     & ' scalar arguments:',/,
     & ' nx           = ',i10,/,
     & ' nz           = ',i10,/,
     & ' dx           = ',e10.4,/,
     & ' dz           = ',e10.4,/,
     & ' xmin         = ',e10.4,/,
     & ' zmin         = ',e10.4,/,
     & ' xs           = ',e10.4,/,
     & ' zs           = ',e10.4,/,
     & ' zd           = ',e10.4,/,
     & ' ap           = ',e10.4)

 1020 format(
     & ' internal scalars:',/,
     & ' size         = ',i10,/,
     & ' ssptr        = ',i10,/,
     & ' apcut_ptr    = ',i10,/,
     & ' work_ptr     = ',i10,/,
     & ' len_work     = ',i10)

      if (idbg.ge.2) then
         write(ipdmp,1030)
         do i=1,nx
            write(ipdmp,1031)i,(v(j+(i-1)*nz),j=1,nz)
         end do
      end if

 1030 format(' input velocity array:')
 1031 format(' trace ',i5,/,6(e10.4,2x))

c idbg >= 2: debug exit

      if (idbg.ge.2) then
         write(ipdmp,*)' TTSOLVE25D ---> bye-bye...'
      end if

c compute squared slowness array, used throughout

      if (idbg.ge.1) then
         write(ipdmp,*)' TTSOLVE25D_INV ---> squared slowness...'
         write(ipdmp,*)' copy to work buffer...'
         write(ipdmp,*)' square...'
         write(ipdmp,*)' reciprocal...'
      end if

      vmin=v(1)

      do i=1,size
         vmin=min(vmin,v(i))
         work(ssptr-1+i)=1.0/(max(v(i),tol)**2)
      end do

      if (vmin.lt.tol) then
         write(ipdmp,*)' Error: TTSOLVE25D_INV'
         write(ipdmp,*)' velocity miniumum = ',vmin,' < ',tol
         write(ipdmp,*)' nz = ',nz,' nx = ',nx
         ier=3
         return
      end if

c compute...

c initialize arrays - call TTINIT25D_INV

      if (idbg.ge.1) then
         write(ipdmp,*)' TTSOLVE25D_INV ---> TTINIT25D_INV..'
      end if
      call ttinit25d_inv(nx,nz,dx,dz,xmin,zmin,xs,zs,zd,ap,
     &     nsz,eik_eps,sincut,tancut,
     &     v,tt,amps,tau_x,tau_z,phi_xr,tau_yy,work(apcut_ptr),
     &     ipdmp,idbg,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: TTSOLVE25D_INV from TTINIT25D_INV'
         return
      end if

c downsweep - call TTAMP25D_INV

      if (idbg.ge.1) then
         write(ipdmp,*)' TTSOLVE25D_INV ---> TTAMP25D_INV'
      end if

      call ttamp25d_inv(nz,nx,dz,dx,nsz,v,
     &	   xmin,xs,zs,zd,
     &     eik_eps,sincut,tancut,work(ssptr),tt,amps,
     &     tau_x,tau_z,phi_xr,tau_yy,work(apcut_ptr),work(work_ptr),
     &     new_work,ipdmp,idbg,ier)

      if (ier.ne.0) then
         write(ipdmp,*)' Error: TTSOLVE25D_INV from TTAMP25D_INV'
         return
      end if

      if (idbg.ge.1) then
         write(ipdmp,*)' TTSOLVE25D_INV ---> SMOOTH_AP'
      end if
      
      call smooth_ap(nz,nx,work(apcut_ptr),work(work_ptr),new_work,
     &     ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: TTSOLVE25D_INV from SMOOTH_AP'
         return
      end if

c ADDED 20.10.95 WWS

      if (idbg.ge.1) then
         write(ipdmp,*)' TTSOLVE25D_INV ---> SCALE_AMP'
      end if      

      call scale_amp_inv(nz,nx,amps,work(apcut_ptr),ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: TTSOLVE25D_INV from SCALE_AMP'
         return
      end if

      call scale_grad_inv(nz,nx,tau_x,tau_z,v,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: TTSOLVE25D_INV from SCALE_GRAD'
         return
      end if

      if (idbg.ge.1) then
         write(ipdmp,*)' TTSOLVE25D_INV: SCALE_GRAD'
      end if

      call bisect_grad_inv(nz,nx,tau_x,tau_z,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: TTSOLVE25D_INV from BISECT_GRAD'
         return
      end if

      return
      end
c=============================================================================





