
c============================ TTAMP25D_FAJ ========================

      subroutine ttamp25d_faj (nz, nx, dz, dx, ntop, 
     &     vel, xmin, xs, zs, zd, eps, ss, tt, amp,
     &     tau_x, tau_z, tau_yy, 
     &     work, lenwork, ipdmp, idbg, ier)
c-------------------------------------------------------------
c    2.5D traveltime and geometric optics amplitude solver
c
c    Driver to extend traveltime and amplitudes throughout
c    a rectangular array, given values on the ntop row (i.e.
c    first index = ntop) and above.
c
c    Uses 2nd order Runge-Kutta time step for numerical 
c    Hamiltonian operator, which returns both the nu-
c    merical Hamiltonian and a suitable right-hand side for
c    the transport equation. The local step is also returned
c    by this operator - it is chosen to be the minimum of a
c    safe CFL step and the remainder of the current step.
c
c    Current version (07.94): right hand sides of both eikonal
c    and transport equations are modified to limit ray incidence
c    angles to an aperture and to damp amplitudes along rays
c    at the aperture boundary.
c
c    WWW             05/93
c    HT              08/93
c    WWS             12/93
c    RV              02/94
c    WWS             07/94 - NOTE THAT EVOLUTION COORDINATE IS
c                            NOW 1ST!!!!
c    KA              03/96 - MODIFIED TO INCLUDE tau_yy and the
c                            out_of_plane component of the 2.5D
c                            amplitude.
c--------------------------------------------------------------

c-- Arguments
      real
     &     dz, dx,              ! steps        
     &     eps,                 ! faithful threshhold
     &     ss(nz,nx),           ! slowness squared
     &     tt(nz,nx),           ! output traveltime array
     &     amp(nz,nx),          ! output amplitude array
     &     tau_x(nz,nx),        ! output x_component of grad tt array
     &     tau_z(nz,nx),        ! output z_component of grad tt array
     &     tau_yy(nz,nx),       ! output yy_component of grad tt array
     &     work(*)              ! workspace

      real
     &     xmin,                ! 
     &     vel(nz,nx),          ! velocity array
     &     x,                   ! x-coordinate of output point on datum
     &     xs,                  ! x-coordinate of the source
     &     zs,                  ! z-coordinate of the source
     &     r,                   ! distance along the ray
     &     zd                   ! depth of datum

      integer
     &     nz, nx, ns,          ! array dimensions
     &     ntop,                ! row index for initial data
     &     lenwork,             ! size of available workspace
     &     ipdmp,               ! dump unit number
     &     ier                  ! error flag

      integer
     &     a_ptr,               ! pointer for a the coefficient of u_x
     &     u_d_ptr,             ! pointer for initial tau_yy at the datum
     &     src_ptr,             ! pointer for the source of the advection
c                                 equation 
     &     u_ptr,                ! pointer for the final tau_yy or
c                                  out_of_plane component of the amplitude
c                                 advection equation or
     &     size,                ! length of array
     &     alpha,               ! stabiltity condition for the solution of
     &     l,                   !
     &     k                    !
c-- Internal Variables
      real
     &     pi,                  ! a Greek letter
     &     root_two,            ! (2.0) to the power of half
     &     dztmp,               ! substep
     &     step_total,          ! partial step left to take
     &     init_amp_peak,       ! amp peak
     &     max_amp,             ! max relative amplitude
     &     min_amp,             ! min relative amplitude
     &     tol                  ! tolerance

      integer
     &     i, i1, j,            ! loop counters
     &     newwork,             ! lenght of remaining workspace for subs
     &     max_partial_steps,   ! guess
     &     idbg,                ! debug flag
     &     idbg_eik,            ! debug flag for eik rhs
     &     idbg_tsp,            ! debug flag for transport rhs
     &     nxp2,                ! nx+2
     &     hamptr,              ! pointer to buffer for rhs of eik eqn
     &     rhsptr,              ! pointer to buffer for rhs of transport eqn
     &     ttptr0,              ! pointer to buffer for travel time
     &     ttptr1,              ! pointer to buffer for travel time
     &     dxtptr,              ! pointer to buffer for travel time x diff
     &     dx2tptr,             ! pointer to buffer for travel time 2nd x diff
     &     apcutptr,            ! pointer to buffer for cutoff
     &     ampptr0,             ! pointer to buffer for amplitude
     &     ampptr1,             ! pointer to buffer for amplitude
     &     work_ptr              ! pointer to buffer for workspace

      integer ntaper

      real dtaper

      logical done, adjust

      data tol /1.e-4/, max_partial_steps /50/, pi /3.1415927/, 
     &     ntaper /4/

c======================================================================

      if (ier.ne.0) return

      idbg_eik=0
      idbg_tsp=0

c set minimum, maximum relative amplitudes

      dtaper=1.0e+00/float(ntaper)
      min_amp=1.e-3
      max_amp=1.e+1

      if (idbg.ge.2) then
         write(ipdmp,*)' TTAMP: initial traveltimes, amplitudes'
         write(ipdmp,*)' tt:'
         write(ipdmp,*)(tt(ntop,j),j=1,nx)
         write(ipdmp,*)' amps:'
         write(ipdmp,*)(amp(ntop,j),j=1,nx)
         write(ipdmp,*)' tau_x:'
         write(ipdmp,*)(tau_x(ntop,j),j=1,nx)
         write(ipdmp,*)' tau_z:'
         write(ipdmp,*)(tau_z(ntop,j),j=1,nx)
         write(ipdmp,*)' tau_yy:'
         write(ipdmp,*)(tau_yy(ntop,j),j=1,nx)
      end if

c--  Grab memory
c----------------
      work_ptr= 1
      call getbuf('work', hamptr, nx, work_ptr, lenwork, ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: TTAMP from GETBUF for hamptr'
         return
      end if
      call getbuf('work', rhsptr, nx, work_ptr, lenwork, ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: TTAMP from GETBUF for rhsptr'
         return
      end if
      call getbuf('work', ttptr0, nx, work_ptr, lenwork, ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: TTAMP from GETBUF for ttptr0'
         return
      end if
      call getbuf('work', ttptr1, nx, work_ptr, lenwork, ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: TTAMP from GETBUF for ttptr1'
         return
      end if
      call getbuf('work', ampptr0, nx, work_ptr, lenwork, ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: TTAMP from GETBUF for ampptr0'
         return
      end if
      call getbuf('work', ampptr1, nx, work_ptr, lenwork, ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: TTAMP from GETBUF for ampptr1'
         return
      end if
      nxp2= nx+ 2
      call getbuf('work', dx2tptr, nxp2, work_ptr, lenwork,
     &     ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*) ' Error: TTAMP from GETBUF for dx2tptr'
         return
      end if
      call getbuf('work', dxtptr, nx, work_ptr, lenwork,
     &     ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*) ' Error: TTAMP from GETBUF for dxtptr'
         return
      end if
      call getbuf('work', apcutptr, nx, work_ptr, lenwork,
     &     ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*) ' Error: TTAMP from GETBUF for dxtptr'
         return
      end if
c===================================================================
c Reserve work space for the calculation of:
c        1. tau_yy
c        2. the out_of_plane component of the amplitude
      size = nx
      call getbuf('work', u_d_ptr, size, work_ptr, lenwork,
     &     ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*) ' Error: TTAMP from GETBUF for U_D_PTR'
         return
      end if

      ns = nz - ntop + 1
      size = nx * ns
      call getbuf('work', a_ptr, size, work_ptr, lenwork,
     &     ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*) ' Error: TTAMP from GETBUF for A_PTR'
         return
      end if
      call getbuf('work', src_ptr, size, work_ptr, lenwork,
     &     ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*) ' Error: TTAMP from GETBUF for SRC_PTR'
         return
      end if
      call getbuf('work', u_ptr, size, work_ptr, lenwork,
     &     ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*) ' Error: TTAMP from GETBUF for U_PTR'
         return
      end if

c===================================================================
      newwork=lenwork- work_ptr

      init_amp_peak=0.0e+00
      do j=1,nx
         init_amp_peak=max(init_amp_peak,amp(ntop,j))
      end do
      min_amp=max(1.0e-10,init_amp_peak*min_amp)
      max_amp=max(1.0e-10,init_amp_peak*max_amp)
CDIR$ NORECURRENCE      
      do j=1,nx
         work(ampptr0-1+j) = log(max(amp(ntop,j),min_amp))
         work(ttptr0-1+j) = tt(ntop,j)
      end do

c--  Debug instructions
c------------------------
      if (idbg.gt.0) then
         write(ipdmp,*) ' TTAMP25D_FAJ: inputs'
         write(ipdmp,*) ' nx, nz, dx, dz '
         write(ipdmp,*)   nx, nz, dx, dz
         write(ipdmp,*) ' eps'
         write(ipdmp,*)   eps
         write(ipdmp,*) ' TTAMP25D_FAJ: pointers'
         write(ipdmp,*) ' ttptr0, ttptr1, ampptr0, ampptr1'
         write(ipdmp,*)   ttptr0, ttptr1, ampptr0, ampptr1
         write(ipdmp,*) ' hamptr, rhsptr, work_ptr'
         write(ipdmp,*)   hamptr, rhsptr, work_ptr
         write(ipdmp,*) ' dxtptr, dx2tptr, apcutptr'
         write(ipdmp,*)   dxtptr, dx2tptr, apcutptr
         write(ipdmp,*) ' u_d_ptr '
         write(ipdmp,*)   u_d_ptr
         write(ipdmp,*) ' a_ptr, src_ptr, u_ptr'
         write(ipdmp,*)   a_ptr, src_ptr, u_ptr
         write(ipdmp,*) ' TTAMP: remaining workspace = ',
     &      newwork
         write(ipdmp,*) ' init_amp_peak = ', init_amp_peak
         write(ipdmp,*) ' min_amp = ',min_amp
         write(ipdmp,*) ' max_amp = ',max_amp
      end if

      min_amp=log(min_amp)
      max_amp=log(max_amp)

      do i= ntop, nz-1

c NOTE that the amplitude work vector contains the natural logarithm
c of the amplitude

c-----------------------------------------               
c--  Set step total to zero. Object is to reach dz
c---------------------------------------------------

         step_total= 0.0
           
         done= .false.

c--  PARTIAL STEP LOOP
c
         do i1= 1, max_partial_steps

            if (.not.done) then

c--  First call to numerical hamiltonian. Returns a
c    suitable value of dztmp each time. dztmp is first
c    initialized to dz at each call with adjust= .true.
c---------------------------------------------------------

               if (idbg.ge.2) then
                  write(ipdmp,*)' TTAMP: step ',i,' substep ',i1
                  write(ipdmp,*)' TTAMP ---> EIK_RHS 1'
               end if

               adjust= .true.
               dztmp= dz
               
               call eikonal_rhs(nx, nz, i, ss,
     &              work(ttptr0), work(hamptr),
     &              work(dxtptr), work(dx2tptr), 
     &              work(apcutptr),
     &              dx, dztmp, dz, step_total, adjust, newwork, 
     &              eps, work(work_ptr), idbg_eik, ipdmp, ier)
      
               if (ier.ne.0) then
                  write(ipdmp,*) ' Error: TTAMP from  EIK_RHS'
                  return
               end if

c--  First Runge-Kutta half-step for tt
c-----------------------------------------

CDIR$ NORECURRENCE
               do j= 1, nx
                  work(ttptr1-1+j) = work(ttptr0-1+j)
     &                 + dztmp* work(hamptr-1+j)
               end do

c If amplitudes are to be computed evaluate RHS in transport equation
c and take first RK half step for amps

               if (idbg.ge.2) then
                  write(ipdmp,*)' TTAMP ---> TRANS_RHS'
               end if

               call transport_rhs(nx, nz, i, ss,
     &              work(ttptr0), work(ampptr0), work(rhsptr),
     &              work(dxtptr), work(dx2tptr), 
     &              work(hamptr), work(apcutptr),
     &              dx, dz, step_total, newwork, 
     &              work(work_ptr), idbg_tsp, ipdmp, ier)

               if (ier.ne.0) then
                  write(ipdmp,*)' Error: TTAMP from TRANS_RHS'
                  return
               end if

CDIR$ NORECURRENCE
               do j= 1, nx
                  work(ampptr1-1+j) = work(ampptr0-1+j)
     &                 + dztmp* work(rhsptr-1+j)
               end do

c--  Second numerical Hamiltonian evaluation
c----------------------------------------------

               if (idbg.ge.2) then
                  write(ipdmp,*) ' TTAMP ---> EIK_RHS 2'
               end if

               step_total= step_total + dztmp
               
               adjust= .false.

               call eikonal_rhs(nx, nz, i, ss,
     &              work(ttptr1), work(hamptr),
     &              work(dxtptr), work(dx2tptr), 
     &              work(apcutptr),
     &              dx, dztmp, dz, step_total, adjust, newwork, 
     &              eps, work(work_ptr), idbg_eik, ipdmp, ier)
               if (ier.ne.0) then
                  write(ipdmp,*) ' Error: TTAMP from  EIK_RHS'
                  return
               end if

c--  Second Runge-Kutta half-step for tt
c-----------------------------------------

CDIR$ NORECURRENCE
               do j= 1, nx
                  work(ttptr0-1+j) = 0.5*
     &                 ( work(ttptr0-1+j) + work(ttptr1-1+j)
     &                 + dztmp* work(hamptr-1+j)  )
               end do

               if (idbg.ge.2) then
                  write(ipdmp,*) 
     &                 ' step          = ',i,
     &                 ' partial step  = ', i1,
     &                 ' step total    = ', step_total,
     &                 ' complement    = ', dz-step_total
               end if

c--  Second evaluation of RHS of transport equation
c----------------------------------------------------
       
               if (idbg.ge.2) then
                  write(ipdmp,*)' TTAMP ---> TRANS_RHS 2'
               end if

               call transport_rhs(nx, nz, i, ss,
     &              work(ttptr1), work(ampptr1), work(rhsptr),
     &              work(dxtptr), work(dx2tptr), 
     &              work(hamptr), work(apcutptr),
     &              dx, dz, step_total, newwork, 
     &              work(work_ptr), idbg_tsp, ipdmp, ier)
                  
               if (ier.ne.0) then
                  write(ipdmp,*)' Error: TTAMP from TRANS_RHS'
                  return
               end if
                  
c--   Second RK half step for amps
                  
CDIR$ NORECURRENCE
               do j= 1, nx
                  work(ampptr0-1+j) = 0.5*(work(ampptr0-1+j)
     &                 +work(ampptr1-1+j)+dztmp* work(rhsptr-1+j))
               end do

c-----------------------------------------               
c--  If successful record output fields
c-----------------------------------------               

               if (abs(dz-step_total).lt.(tol*dz)) then
                  done= .true.
CDIR$ NORECURRENCE
                  do j= 1, nx
                     tt(i+1,j) = work(ttptr0-1+j)
                     tau_x(i+1,j) = work(dxtptr-1+j)
                     tau_z(i+1,j) = work(hamptr-1+j)
                     work(ampptr0-1+j)=max(min_amp,min(max_amp,
     &                    work(ampptr0-1+j)))
                     amp(i+1,j)= exp(work(ampptr0-1+j))
                  end do
c taper!
CDIR$ NORECURRENCE
                  do j=0,ntaper
                     amp(i+1,1+j)=dtaper*j*amp(i+1,1+j)
                     amp(i+1,nx-j)=dtaper*j*amp(i+1,nx-j)
                  end do

               end if

            end if
            
         end do

c--  If we're still not done, then we failed to find an ok step
c----------------------------------------------------------------

         if (.not.done) then
            write(ipdmp,*) ' Error:TTAMP'
            write(ipdmp,*) ' after ', max_partial_steps, ' steps'
            write(ipdmp,*) ' failed to take step ', i
            ier= 187
            return
         end if

c DEBUG!!!!!!!!!!!!!!!!!!!!!! - stop at the first step

         if (idbg.ge.5) then
            if (i.gt.ntop) then
               write(ipdmp,*)' DEBUG EXIT !!!!!!!'
               ier=555
               return
            end if
         end if

      end do

c==================================================================
c Next prepare the input data for the numerical solver of the 
c advection equations to compute tau_yy and the out_of_plane
c component of the amplitude.
c
c           u_z + au_x = 1/tau_z
c                  
c             u(x,zd) =  work(u_d_ptr) 
c
c Here                  u = 1/tau_yy
c                       u(x,zd)=u at the datum
c----------------------------------------------------------------------
c           u_z + au_x = -tau_yy/(2.0*tau_z)
c                  
c             u(x,zd) =  work(v_d_ptr) 
c
c Here                  u = out-of-plane component of the amplitude
c                       u(x,zd)=w at the datum
c*******************************************************************************
c
c Initialize stability condition:
c
c      
      alpha = 0
      root_two = 1.0/sqrt(2.0)

CDIR$ NORECURRENCE
      do i = 1,nx
c
c calculate u(x,zd)
c
         x = xmin + (i - 1) * dx
         r = sqrt( (x-xs)*(x-xs) + (zd-zs)*(zd-zs) )
c
c compute tau_yy and out-of-plane component of amplitude at the datum:
c
         work(u_d_ptr + i-1) = vel(ntop,i) * r 
c
c For each input point inside the model calculate
c       1. the 2-D array a:
c       2. the 2-D array src:
c
c
CDIR$ NORECURRENCE
         do j= 1,ns
            l = j + ntop -1
            k = (i-1)*ns + j-1
            work(a_ptr+k) = tau_x(l,i)/tau_z(l,i)
            alpha = max ( abs(nint(work(a_ptr+k))),alpha )
            work(src_ptr+k) = 1.0e+00/tau_z(l,i)
         end do
      end do
c
c To get tau_yy
c
      call sigma_global(nx,ns,dx,dz,
     &              work(a_ptr),work(src_ptr),work(u_ptr),
     &              work(u_d_ptr),alpha,work,work_ptr,
     &              newwork,ipdmp,idbg,ier)

      if (ier.ne.0) then
         write(ipdmp,*)' Error: TTAMP25D_FAJ from SIGMA_GLOBAL'
         return
      end if
c
c Return tau_yy and compute out-of-plane component of amplitude
c        using tau_yy:
c
      do i = 1,nx
         do j= 1,ns
	    l = j + ntop -1
            k = (i-1)*ns + j-1
	    tau_yy(l,i) = 1.0e+00/work(u_ptr + k)
	    amp(l,i) = amp(l,i)*sqrt( tau_yy(l,i) )*root_two
         end do
      end do
c
      if (idbg.ge.2) then
         write(ipdmp,*)' TTAMP25D_FAJ: return'
      end if

      return
      end
c============================================================================
