c============================ READ_TTDATA ================================
      subroutine read_ttdata3d(cache,inversion,three_d,
     &     velmodel,tt,work,token,
     &     velnz,velzorig,veldz,velnx,velxorig,veldx,
     &     seismnt,seismdt,n1box,n2box,o2box,
     &     srcdep,srcpos,save_ttflag,tt_sam,tt_size,
     &     cv_flag,update,lenwork,
     &     ap,zd,nfls,ipdmp,idbg,ier)

c RV      07.93
c rev WWS 11.94
c rev WWS 11.95
c rev WWS 07.96
c rev WWS 09.96

c determines grid on which traveltimes will be computed and Kirchhoff
c summation performed (n1box, n2box, o2box).
c reads or creates traveltime data for a source centered horizonatally
c at location (srcdep,srcpos) in the n1box x n2box grid at (horizontal)
c offset o2box. Assumes homogeneous medium to
c depth zd, cuts off traveltime and amplitude computation
c outside of aperture ap. Does actual computation on resampled
c grid with resample factor tt_sam. Manages list of already computed
c source locations and saved files in static local arrays
c (srcloc, srcfile). If save_ttflag = 1 saves computed tt/amp fields
c on decimated grid to file. If update = 0 uses any tt/amp tables
c found in files with correct location; else computes them.

c note that the parameter nfls is assumed to be initialized in the calling
c routine - it should be reset to 0 at the beginning of a new simulation
c for which update > 0. This is VERY IMPORTANT, as it controls both the
c computation of the grid on which traveltimes and amplitudes will be
c computed and the generation of the filename list for saved tt/amp tables.

c SO: if nfls = 0, 1) determine n1box, n2box, o2box 2) start
c re-writing the tt/amp files if save_ttflag > 0 several steps are
c repeated each time even though they should give the same results,
c because they simply don't cost enough to make saving the results
c worthwhile. These include computing n1box_s and n2box_s, and
c regenerating the source location and filename lists.

c MAJOR REVISION OF 11.95: 
c
c implements the "core-disk-solver" hierarchy by adding an incore buffer
c in which to cache ttamp tables. avoids data movement by using this
c buffer as output space and passing back an offset.
c
c Calling routine passes a buffer tt of size tt_size used to cache 
c coarse grid ttamp tables until it overflows,
c then one or more of the oldest tables is/are kicked out until enough
c room is made to store the new table. In the current design, the size of
c the tt box is independent of source location, so all tables have the same
c size so this works perfectly. In the future this mess should be rewritten
c in C.
c
c The tt buffer is divided implicitly into segements by the token input.
c New tt tables only replace old ones with the same token.
c
c Note that header information is essential in properly computing the
c fine grid ttamp and the geometric information (n1box, n2box, o2box).
c Therefore the tt buffer contains partial header information in the
c following form:
c
c    word      contents
c    1         float(n1box_s)
c    2         float(n2box_s)
c    3         o1box_s
c    4         o2box_s
c    5         d1box_s
c    6         d2box_s
c    7...      samples of coarse grid tables
c
c Logic of this version: on call,
c  - if cache flag is not set, reset all table variables to 
c    default values, set flag
c  - open and read position file if present. contains
c       nfls     - number of table computed so far (= 0 for new velocity)
c       nfls_incore
c                - number stored in core
c       srcloc   - coordinates of source
c       srcoff   - offset into buffer for this table (= -1 if table is not
c                  in buffer)
c       srcage   - age flag - top value is nfls_incore, incremented each
c                  time a new table is added to the buffer
c  - check for presence of tables in tt buffer, signified by
c    srcage > 0. if table present read header words
c    presence demands only same source point coordinates
c  - if not check disk files. if table is found on disk, secure
c    space in buffer if necessary by bumping oldest table with 
c    same token, then read and interpolate into buffer.
c  - if table is not found on disk, or if the new flag is set,
c    call eikonal/transport solver. If tables are to be saved
c    to disk, do so. Then proceed as in previous step.
c  - at this point have coarse grid tables. Interpolate then
c    return 
c
c WWS 16.10.95 - inversion variable formerly used to decide whether to return
c full set of tables - now want all of them always, so make it local and
c fix it in a data statement.

c WWS 20.10.95 - now return
c   1) amplitudes multiplied by cutoff for adjoint
c   2) RECIPROCAL amplitudes multiplied by cutoff for inversion
c therefore need inversion flag again

c WWS 16.10.95: assume that one always wants all tables or at least the apcut
c WWS 20.10.95: unassume this

c INPUT ARGUMENTS:

      integer inversion         ! true if you want to use Beylkin Method
c                                 false otherwise                              
      integer three_d           ! true if you want to use 2.5D amplitude
c                                 false otherwise                              
      integer
     &     ier,                 ! error flag
     &     ipdmp,               ! dump unit
     &     cache,               ! flag for cache initialization
     &     lenwork,             ! length of work array
     &     tt_sam,              ! resample factor (1,2 or 4)
     &     tt_size,             ! size of cache
     &     token,               ! identifier - used to mark subsets of
c                                 tables to replace selectively 
     &     cv_flag,              ! amplitude option
     &     velnz,               ! first dimension of velocity model
     &     velnx,               ! second dimension of velocity model
     &     seismnt,             ! number of time samples in trace
     &     save_ttflag,         ! flag indicating whether the traveltimes 
c                                 should be saved to disk
     &     update               ! flag indicating whether new maps should 
c                                 be calculated EVEN if they exist

      real 
     &     srcpos,              ! source offset
     &     srcdep,              ! source depth
     &     velzorig,            ! coord of 1st sample, dim1, vel model
     &     velxorig,            ! coord of 1st sample, dim2, vel model
     &     veldz,               ! sample rate, dim1, vel model
     &     veldx,               ! sample rate, dim2, vel model
     &     seismdt,             ! sample rate in time, seismogram
     &     ap,                  ! aperture for amplitude cutoff
     &     zd                   ! datum depth - vel is const above this

c     velocity model
      real
     &     velmodel(velnz,velnx)

c     workspace buffers
      real
     &     tt(tt_size),
     &     work(lenwork)

c OUTPUT ARGUMENTS:

      integer
     &     n1box,               ! first dimension of tt/amp/tau_x/tau_z box
     &     n2box,               ! second dimension of tt/amp/tau_x/tau_z box
     &     ttamp_ptr            ! pointer to array containing tt in first part,
c                                 amp in second part, tau_x in third part
c                                 tau_z in fourth and phi_xr fifth part
c                                 and apcut in the sixth part
      real
     &     o2box                ! offset of left edge of
c                                 tt/amp/tau_x/tau_z/phi_xr box

c INTERNAL VARIABLES:

      integer 
     &     n1box_s,             ! 1st dimension of coarse grid tt/amp/tau_x/tau_z box
     &     n2box_s,             ! 2nd dimension of coarse grid tt/amp/tau_x/tau_z box
     &     o2box_index,         ! index of left side of tt/amp/tau_x/tau_z box
     &     lenwork_loc,         ! workspace remaining after local needs
     &     work_ptr,             ! index to beginning of free workspace
     &     rw,                  ! read/write flag
     &     sindex,              ! source index
     &     oldest,              ! index of oldest table in cache
     &     oldestage,           ! age of oldest table in cache
     &     i,j,iv,jv,           ! loop counters
     &     ivmin,jvmin,         ! corner indices of velocity grid
     &     idbg                 ! debug flag

      integer ntables

      real
     &     o1box_s,             ! 1st dim origin of coarse grid
c                                 tt/amp/tau_x/tau_z/phi_xr box
     &     o2box_s,             ! 2nd dim origin of coarse grid
c                                 tt/amp/tau_x/tau_z/phi_xr box
     &     d1box_s,             ! 1st dim sample rate of coarse grid
c                                 tt/amp/tau_x/tau_z/phi_xr box
     &     d2box_s,             ! 2nd dim sample rate of coarse grid
c                                 tt/amp/tau_x/tau_z/phi_xr box
     &     srcpos_box           ! rel offset of src in its box

      integer
     &     ttamp_sptr,          ! pointer to seg of work holding resampled
c                                 tt/amp/tau_x/tau_z/tau_xx_xr
     &     vel_sptr             ! pointer to seg of work holding resampled vel

      integer 
     &     next,                ! pointer to next free word in work buffer
     &     size                 ! length of segment to grab out of work

      integer
     &     nfls,                ! number of saved traveltime tables
     &     nfls_incore,         ! number saved incore
     &     top_incore,          ! number of words saved incore
     &     mxfls                ! maximum number of sources & receivers
      parameter (mxfls=2000)

      integer
     &     srcoff(mxfls),       ! offset of table in tt buffer
c                                 set = 0 when table is not present in core
     &     srcage(mxfls),       ! age parameter - used in "oldest out" rule
c                                 set = 0 when table is not present in core
c                                 always started at 1, so oldest table always
c                                 has srcage = nfls_incore
     &     srctoken(mxfls)      ! token - used to identify subsets of cache
      real srcloc(mxfls,2)      ! coordinates of source positions already
c                                 visited in this simulation
      character*80 
     &     srcfile(mxfls)       ! filenames for saved tt/amp arrays

      logical new(mxfls)        ! true if model is new and tt/amp arrays
c                                 must be recomputed

c      static 
      save
     &     nfls_incore,top_incore,
     &     srcloc, srcfile, new,
     &     srcage, srctoken,
     &     srcoff               ! these should be saved from call to call

      real test,tol             ! used to determine whether current source
c                                 location has already been visited

      real 
     &     slnsavg,             ! slowness average - updated for each
c                                 new velocity model, used for computation
c                                 of traveltime box 
     &     maxtime,             ! time limit
     &     maxoff,              ! max offset = sin(ap)*max depth
     &     maxdep,              ! max depth reached at average slowness
     &     pi,                  ! uh... let's see ...
     &     fudge                ! factor to give margin for deviation from 
c                                 average slowness - weak spot in this design

      save slnsavg            ! keep this around so that it only needs computing
c                                 at beginning of new sim with new vel

      logical found             ! flag for search on source location

      character*4 
     &     ftyp,                ! format designator for tt/amp files
     &     ext4                 ! 4 digit character rep of file number

      character*3 ext3          ! 3 digit character rep of file number
      character*2 ext2          ! 2 digit character rep of file number
      character*1 ext1          ! 1 digit character rep of file number
      character*1 blank

      character charbuf(80)     ! character array in which to store 
c                                 c-compatible string

      integer ntotal            ! workspace for total words read/written
c                                 to disk

      data tol /1.0e-2/
      data fudge /1.2e+00/, pi /3.1415927e+00/
      data blank /' '/

c------------------------------------------------------

      if (ier.ne.0) return
      if (idbg.gt.0) then
         write(ipdmp,*)' Entering READ_TTDATA'
         write(ipdmp,*)' workspace available = ',lenwork
      endif

c set file format for saved ttamp tables
      
      ftyp='.sep'

c set up reads, correct number of tables per file
    
      rw=1

      if((inversion.eq.1).and.(three_d.eq.1))then
	 ntables = 6
      else if((inversion.eq.1).and.(three_d.eq.0)) then
	 ntables = 5
      else if((inversion.eq.0).and.(three_d.eq.1)) then
	 ntables = 5
      else
	 ntables = 2
      endif

c initialize the pointer to free workspace, save length of
c work vector

      lenwork_loc = lenwork

c I. check to see whether position is on list of those
c for which data is held incore or on disk.
          
c this is a new simulation with a new velocity if nfls = 0 and
c update = 1. In that case reset the new array to .true. throughout.
c 12.11.95 - also the various buffer arrays

      if ((nfls.eq.0).and.(update.eq.1)) then
         do i=1,mxfls
            new(i)=.true.
            srcoff(i)=0
            srcage(i)=0
            srctoken(i)=-1
            nfls_incore=0
            top_incore=0
         end do
         if (idbg.ne.0) then
            write(ipdmp,*)' READ_TT: beginning new simulation with'
            write(ipdmp,*)' new velocity; recompute all tt/amp tables'
         end if
      end if

c if this is the first call to READ_TT in computing a block of gathers,
c the cache flag should be unset. Then set all of the buffer arrays to 
c their default values, signifying that no traveltime tables are stored
c in core. This is necessary because the integrity of the buffer is not
c guaranteed between calls to the routines which call READ_TT, as it is
c just a part of the work buffer set aside by the calling routines.

c MAJOR REDESIGN, 19.08.96: with new main routine, the ttamp buffer is now
c passed from above the routines which call READ_TT, so now the integrity
c of this buffer IS (or can be) guaranteed between calls. Thus there is no
c need anymore to write ttamp tables when they are cached - they can be 
c written only when bumped.

c the calling routine (either main or operator) will run through a simulation -
c during this the memory is safe. However when the simulation is over the
c memory may be released depending on how the calling routine handles it -
c therefore to be safe the cache flag should be reset when a new simulation
c is started, even if the update flag has not been set. This is done by 
c setting cache_flag = 0 when irecord = 1 in the driver routine calling
c READ_TT.

      if (cache.eq.0) then
         if (idbg.gt.0) then
            write(ipdmp,*)' READ_TT: cache flag = ',cache
            write(ipdmp,*)' so reinitialize cache tables'
         end if
         do i=1,mxfls
            srcoff(i)=0
            srcage(i)=0
            srctoken(i)=-1
            nfls_incore=0
            top_incore=0
         end do
         if (idbg.ne.0) then
            write(ipdmp,*)' READ_TT: beginning new gather computation'
            write(ipdmp,*)' with no ttamp tables stored in core'
         end if         
      end if

c check whether the source location has been visited before in this
c simulation. Note that if nfls = 0, signalling a new velocity model
c and new simulation, this is a null loop leaving sindex=0

      sindex=0
      if(save_ttflag.ne.0) then
         do i=1,nfls
            test=abs(srcpos-srcloc(i,2))+abs(srcdep-srcloc(i,1))
            if (test.lt.tol) then
               sindex=i
               if (idbg.gt.0) then
                  write(ipdmp,*)' READ_TT:'
                  write(ipdmp,*)' found current source'
                  write(ipdmp,*)' coordinates ',srcdep,', ',srcpos
                  write(ipdmp,*)' match saved coordinates ',
     &                 srcloc(sindex,1),', ',srcloc(sindex,2)
                  write(ipdmp,*)' index of saved source = ',sindex
               end if
            end if
         end do
         if ((sindex.eq.0).and.(idbg.gt.0)) then
            if (nfls.gt.0) then
               write(ipdmp,*)' READ_TT: '
               write(ipdmp,*)' did not find current source'
               write(ipdmp,*)' coordinates ',srcdep,', ',srcpos
               write(ipdmp,*)' in list of those already computed'
               write(ipdmp,*)' of which there are ',nfls
            else 
               write(ipdmp,*)' READ_TT:'
               write(ipdmp,*)' begin new sim with new velo:
     &	                       nfls = ',nfls
            end if
         end if
      end if

c II. if source position has been visited, then determine whether 
c ttamp table resides in core. This is signified by a positive
c srcage. If so then read tables from core.

      found=.false.

      if ((sindex.gt.0).and.(srcage(sindex).ne.0)) then
         found=.true.
         n1box_s=nint(tt(srcoff(sindex)))
         n2box_s=nint(tt(srcoff(sindex)+1))
         o1box_s=tt(srcoff(sindex)+2)
         o2box_s=tt(srcoff(sindex)+3)
         d1box_s=tt(srcoff(sindex)+4)
         d2box_s=tt(srcoff(sindex)+5)
         if (idbg.gt.0) then
            write(ipdmp,*)' READ_TT:'
            write(ipdmp,*)' found table ',sindex,' in core'
            write(ipdmp,*)' offset     = ',srcoff(sindex)
            write(ipdmp,*)' age        = ',srcage(sindex)
            write(ipdmp,*)' n1box_s    = ',n1box_s
            write(ipdmp,*)' n2box_s    = ',n2box_s
            write(ipdmp,*)' o1box_s    = ',o1box_s
            write(ipdmp,*)' o2box_s    = ',o2box_s
            write(ipdmp,*)' d1box_s    = ',d1box_s
            write(ipdmp,*)' d2box_s    = ',d2box_s
         endif

c if we get to this point then at least one ttamp table has been
c computed, but it's not saved in core. So look on disk.
 
c if the source position has been visited check to see whether the
c corresponding ttamp file has been written

      else if ((sindex.gt.0).and.(srcage(sindex).eq.0)) then         
         inquire(file=srcfile(sindex),exist=found)
         if (ier.ne.0) then
            write(ipdmp,*)' Error: READ_TT from FIND_FILE'
            return
         end if
         if (idbg.gt.0) then
            if (found) then
               write(ipdmp,*)' the tt/amp tables are saved in ',
     &              srcfile(sindex)
               if (new(sindex)) then
                  write(ipdmp,*)' however the update flag is set '
                  write(ipdmp,*)' so the tables will be recomputed'
               else
                  write(ipdmp,*)' these will now be read and used'
               end if
            else
               write(ipdmp,*)' the file ',srcfile(sindex),' does not ',
     &              'exist'
               write(ipdmp,*)' so the tt/amp table must be computed'
               write(ipdmp,*)' the save flag value is ',save_ttflag
            end if
         end if
      end if

c     if 
c        (1) the ttamp table is NOT stored incore, and
c        (2) it has been computed (not new), and
c        (3) the file containing it has been found,
c     read the ttamp coarse grid geometry information from the headers.
c
      if ((found).and.(.not.new(sindex)).and.(srcage(sindex).eq.0)) 
     &     then

c     create a C-readable string to pass as the filename

         size=index(srcfile(sindex),blank) - 1
         do i=1,size
            charbuf(i)=srcfile(sindex)(i:i)
         end do
         charbuf(size+1)=char(0)

c     read coarse grid traveltime temporarily into the beginning
c     of the work array, also grid parameters

         ntotal=lenwork_loc
         call readsep(charbuf,n1box_s,n2box_s,
     &        d1box_s,d2box_s,o1box_s,o2box_s,
     &        work,ntotal,idbg,ier)
         if (ier.ne.0) then
            write(ipdmp,*)' Error: READ_TT from READSEP'
            return
         end if
         if (idbg.gt.2) then
            write(ipdmp,*)' Return: READ_TT from READSEP'
            write(ipdmp,*)' read ',ntotal,' floats'
         endif

c note that Roelof stored the traveltime and amplitude tables
c together in one record. Now other quantities may also be stored
c in each file, i.e. a total of ntables arrays. Since the storage 
c is in column order, the number of traces is really
c ntables*n2box_s - so divide by ntables, checking that you really get
c what you are supposed to get

         if (n2box_s.ne.(ntables*(n2box_s/ntables))) then
            write(ipdmp,*)' Error: READ_TT'
            write(ipdmp,*)' tt/amp record must have even '
            write(ipdmp,*)' number of traces, but = ',n2box_s
            write(ipdmp,*)' something wrong with write of '
            write(ipdmp,*)' tt/amp data'
            ier=1
            return
         end if

         n2box_s=n2box_s/ntables

         if (idbg.gt.0) then
            write(ipdmp,*)' READ_TT: box geometry'
            write(ipdmp,*)' coarse grid parameters read from file'
            write(ipdmp,*)' n1box_s     = ',n1box_s
            write(ipdmp,*)' n2box_s     = ',n2box_s
            write(ipdmp,*)' d1box_s     = ',d1box_s
            write(ipdmp,*)' d2box_s     = ',d2box_s
            write(ipdmp,*)' o1box_s     = ',o1box_s
            write(ipdmp,*)' o2box_s     = ',o2box_s
         end if

      end if

c if table exists (either in core or on disk) access it

      if (sindex.gt.0) then

c     define fine grid parameters, and check that geometry is correct, 
c     relative to global geometry
         
         n1box = 1+tt_sam*(n1box_s-1)
         n2box = 1+tt_sam*(n2box_s-1)
         o2box = o2box_s

         if (abs(d1box_s - tt_sam*veldz).gt.tol*veldz) then
            write(ipdmp,*)' Error: READ_TT'
            write(ipdmp,*)' geometry of tt/amp file read from file'
            write(ipdmp,*)' incompatible with global geometry'
            write(ipdmp,*)' file coarse grid dz = ',d1box_s
            write(ipdmp,*)' resample rate       = ',tt_sam
            write(ipdmp,*)' fine grid dz        = ',veldz
            write(ipdmp,*)' coarse grid dz      = ',veldz*tt_sam
            ier=1
            return
         end if

         if (abs(d2box_s - tt_sam*veldx).gt.tol*veldx) then
            write(ipdmp,*)' Error: READ_TT'
            write(ipdmp,*)' geometry of tt/amp file read from file'
            write(ipdmp,*)' incompatible with global geometry'
            write(ipdmp,*)' file coarse grid dx = ',d2box_s
            write(ipdmp,*)' resample rate       = ',tt_sam
            write(ipdmp,*)' fine grid dx        = ',veldx
            write(ipdmp,*)' coarse grid dx      = ',veldx*tt_sam
            ier=1
            return
         end if

c     revision of 230996: this check no longer necessary

c         if (o1box_s.ne.velzorig) then
c            write(ipdmp,*)' Error: READ_TT'
c            write(ipdmp,*)' file depth origin = ',o1box_s
c            write(ipdmp,*)' disagrees with global depth origin'
c            write(ipdmp,*)' = ',velzorig
c            ier=1
c            return
c         end if

c     set up workspace to hold fine and coarse grid traveltime, amplitudes,
c     tau_x, tau_z and phi_xr  note that ttamp_ptr will be = 1 in all cases,
c     so that the first segment of work is filled with the
c     tt/amp/tau_x/tau_z/phi_xr data on return

         next=1
         size=ntables*n1box*n2box

         if (idbg.gt.0) then
            write(ipdmp,*)' setting up workspace for ttamp grids'
            write(ipdmp,*)' lenwork = ',lenwork
         endif

         call getbuf('work',ttamp_ptr,size,next,lenwork,ipdmp,ier)
         if (ier.ne.0) then
            write(ipdmp,*)' Error: READ_TT from GETBUF for ttamp_ptr'
            return
         end if

         size=ntables*n1box_s*n2box_s

         call getbuf('work',ttamp_sptr,size,next,lenwork,ipdmp,ier)
         if (ier.ne.0) then
            write(ipdmp,*)
     &           ' Error: READ_TT from GETBUF for ttamp_sptr'
            return
         end if
            
         if (idbg.gt.0) then
            write(ipdmp,*)' READ_TT: pointers'
            write(ipdmp,*)' ttamp_ptr         = ',ttamp_ptr
            write(ipdmp,*)' ttamp_sptr        = ',ttamp_sptr
         end if
            
         work_ptr=ttamp_sptr

c if data must be read from disk (srcage=0) do so; if not it's already
c in the tt buffer.

         if (srcage(sindex).eq.0) then

c in the current design it has already been read from disk - it's at
c the beginning of the work buffer. So copy it to appropriate place.

            i=1
            size=ntables*n1box_s*n2box_s
            call scopy(size,work,i,work(ttamp_sptr),i)
            if (idbg.gt.0) then
               write(ipdmp,*)' read ttamp table ',sindex
               write(ipdmp,*)' from disk file ',srcfile(sindex)
            end if

c     note that work_ptr and lenwork are updated

c otherwise the traveltimes need to be copied from the cache
c [A LOT of copying could be avoided if this stuff were rewritten in
c  C or C++...]

         else

            if (idbg.ne.0) then
               write(ipdmp,*)
     &              ' copying coarse grid ttamp table from cache'
            end if
            i=1
            size=ntables*n1box_s*n2box_s
            call scopy(size,tt(srcoff(sindex)+6),i,work(ttamp_sptr),i)

         end if
            
      else

c     if nfls = 0, signalling the beginning of a new simulation with
c     a new velocity model: compute new average slowness for 
c     tt/amp box definition

c     3.1.95: also do this computation if the current ttamp file
c     is not found

         if (idbg.gt.0) then
            write(ipdmp,*)' READ_TT: compute average slowness'
         end if
         if ((nfls.eq.0).or.(.not.found)) then
            slnsavg=0.0e+00
            do i=1,velnz
               do j=1,velnx
                  slnsavg=slnsavg + 1.0/velmodel(i,j)     
               end do
            end do
            slnsavg=slnsavg/(velnz*velnx)
            
         end if   

         if (idbg.gt.0) then
            write(ipdmp,*)' READ_TT: slnsavg = ',slnsavg
         end if
         
c     compute one-way time to deepest possible reflection point
c     at average slowness. Use this plus aperture to compute
c     max offset. limit depth to deepest point in model.

c     revision of 230996: DON'T limit depth to deepest point in (velocity)
c     model
c     assumes minimum depth = 0 (but not min depth in velocity model!)

         maxtime=0.5*(seismnt-1)*seismdt
c         maxdep=min((maxtime/slnsavg),
c     &        velzorig+veldz*(velnz-1))
         maxdep=maxtime/slnsavg         
         maxoff=maxdep*sin(ap*pi/180.0e+00)
         
c     note that n2box is defined so as always to be odd
c     also define n1box so that after adjustment for resampling
c     will guaranteed be less that velnz
         
c     revision of 230996: no longer restrict n1box by velnz

c         n1box=min(nint((srcdep+maxdep)/veldz),velnz-tt_sam)+1
         n1box=nint((srcdep+maxdep)/veldz)+1
         n2box=2*nint(fudge*maxoff/veldx)+1
         
c     make max offset and source position integer multiples of veldx,
c     use to define o2box = left edge of tt/amp box
         
         maxoff=(n2box-1)*veldx*.5
         o2box=int(srcpos/veldx)*veldx - maxoff
         
         if (idbg.gt.0) then
            write(ipdmp,*)' READ_TT: computation of tt/amp box'
            write(ipdmp,*)' nfls:                    ',nfls
            write(ipdmp,*)' average slowness:        ',slnsavg
            write(ipdmp,*)' maximum time:            ',maxtime
            write(ipdmp,*)' maximum depth range:     ',
     &           maxdep
            write(ipdmp,*)' maximum offset:          ',maxoff
            write(ipdmp,*)' n1box:                   ',n1box
            write(ipdmp,*)' n2box:                   ',n2box
            write(ipdmp,*)' o2box:                   ',o2box
         end if

c     revision of 230996: tt computation box no longer need be subset of
c     velocity model domain

c         if (o2box.lt.velxorig) then
c            write(ipdmp,*)' Error: left edge of tt/amp box'
c            write(ipdmp,*)' transgresses model'
c            write(ipdmp,*)' left edge of tt/amp box   :',o2box
c            write(ipdmp,*)' left edge of model        :',velxorig
c            ier=1
c            return
c         end if

c         if (o2box+(n2box-1)*veldx.gt.
c     &        velxorig+(velnx-1)*veldx) then
c            write(ipdmp,*)' Error: right edge of tt/amp box'
c            write(ipdmp,*)' transgresses model'
c            write(ipdmp,*)' right edge of tt/amp box   :',
c     &           o2box+(n2box-1)*veldx
c            write(ipdmp,*)' right edge of model        :',
c     &           velxorig+(velnx-1)*veldx
c            ier=1
c            return
c         end if
         
c     define the resampled grid, if necessary redefine the n1box and n2box
c     dimensions
         
         if (tt_sam.lt.1) then
            write(ipdmp,*)' Error: READ_TT'
            write(ipdmp,*)' resample rate = ',tt_sam,' should be >= 1'
            ier=9
            return
         end if
         
         n1box_s=1+int(float(n1box-1)/float(tt_sam))
         n1box=1+(n1box_s-1)*tt_sam
         n2box_s=1+int(float(n2box-1)/float(tt_sam))
         n2box=1+(n2box_s-1)*tt_sam

         if (idbg.gt.0) then
            write(ipdmp,*)' resample rate ',tt_sam
            write(ipdmp,*)' revised n1box ',n1box
            write(ipdmp,*)' revised n2box ',n2box
            write(ipdmp,*)' n1box_s       ',n1box_s
            write(ipdmp,*)' n2box_s       ',n2box_s
         end if
         
c     note that the offset of the left edge (o2box) remains the same.
c     revision of 230996: o1box_s is always = 0

c         o1box_s=velzorig
         o1box_s=0.0e+00
         o2box_s=o2box

c     sample rates

         d1box_s=veldz*tt_sam
         d2box_s=veldx*tt_sam
         
c     check bottom and right side of box for legitimacy
c     revision of 230996: these checks no longer needed

c         if (n1box.gt.velnz) then
c            write(ipdmp,*)' Error: READ_TT'
c            write(ipdmp,*)' too many samples vertically in table'
c            write(ipdmp,*)' = ',n1box,' more than vertical samples'
c            write(ipdmp,*)' in model = ',velnz
c            ier=1
c            return
c         end if

c         if (n2box+nint((o2box-velxorig)/veldx).gt.velnx) then
c            write(ipdmp,*)' Error: READ_TT'
c            write(ipdmp,*)' right side of box has global index ='
c            write(ipdmp,*)n2box+nint((o2box-velxorig)/veldx)
c            write(ipdmp,*)' this exceeds the max horizontal index of'
c            write(ipdmp,*)' model = ',velnx
c         end if

c     workspace allocation - finite difference computation

         next=1

c     set up space to hold fine grid ttamp tables - must be done
c     here to guarantee that this is the first segment of work on 
c     return

         if (idbg.gt.0) then
            write(ipdmp,*)' setting up workspace for ttamp grids'
            write(ipdmp,*)' lenwork = ',lenwork
         endif

         size=ntables*n1box*n2box
      
         call getbuf('work',ttamp_ptr,size,next,lenwork,ipdmp,ier)
         if (ier.ne.0) then
            write(ipdmp,*)' Error: READ_TT from GETBUF for ttamp_ptr'
            return
         end if

c     set up workspace to hold coarse grid traveltime,
c     amplitudes, x_component, z_component of gradient of traveltime
c     and mixed x, x_r derivative of traveltime.
         
         size=ntables*n1box_s*n2box_s
         
         call getbuf('work',ttamp_sptr,size,next,lenwork,ipdmp,ier)
         if (ier.ne.0) then
            write(ipdmp,*)' Error: READ_TT from GETBUF for ttamp_sptr'
            return
         end if
         
c     to create source: resample velocity to coarser grid, compute tt/amp
c     on coarse grid, save to file if indicated, then interpolate back
c     to original grid
         
         o2box_index = int(0.01+(o2box/veldx))+1
         
         size = n1box_s*n2box_s
         
         call getbuf('work',vel_sptr,size,next,lenwork,ipdmp,ier)
         if (ier.ne.0) then
            write(ipdmp,*)' Error: READ_TT from GETBUF for vel_sptr'
            return
         end if
 
c     revision of 230996, real meat: here you sample the velocity, not
c     on its own grid, but possibly outside - the velocity gets extended
c     by constants, first vertically then horizontally

c     these give the upper left corner indices of the velocity model in 
c     the reflectivity grid, assuming (as we still do for the moment) that
c     the grid spacings are the same.

         ivmin=1+nint(velxorig/veldx)
         jvmin=1+nint(velzorig/veldz)

         do i=1,n2box_s
            iv=1-ivmin+(i-1)*tt_sam
            iv=max(1,iv)
            iv=min(velnx,iv)
            do j=1,n1box_s
               jv=1-jvmin+(j-1)*tt_sam
               jv=max(1,jv)
               jv=min(velnz,jv)
               work(vel_sptr-1+(i-1)*n1box_s+j) = velmodel(jv,iv)
            end do
         end do
         
         if (idbg.gt.0) then
            write(ipdmp,*)' READ_TT: pointers'
            write(ipdmp,*)' ttamp_ptr        = ',ttamp_ptr
            write(ipdmp,*)' ttamp_sptr       = ',ttamp_sptr
            write(ipdmp,*)' vel_sptr        = ',vel_sptr
         end if

c     update lenwork_loc - note that original length of work vector is
c     still in lenwork
         
         lenwork_loc=lenwork_loc-next
         
c     in the call to stt, note that velzorig is assumed to have stayed
c     the same in the restriction to the box, whereas velxorig has been
c     shifted to o2box. So the relative srcpos has shifted but the
c     relative srcdep has not.
         
         srcpos_box=srcpos-o2box
         
         call stt_create(inversion,three_d,ntables,
     &        n1box_s,o1box_s,d1box_s,
     &        n2box_s,o2box_s,d2box_s,seismdt,
     &        srcpos,srcdep,
     &        work(vel_sptr),cv_flag,work(ttamp_sptr),
     &        work(next),lenwork_loc,ap,zd,ipdmp,idbg,ier)
         if (ier.ne.0) then
            write(ipdmp,*)' Error: READ_TT from STT_CREATE  '
            return
         end if
         
c     ALWAYS update nfls and the srcloc and srcfiles arrays, even
c     if save_ttflag = 0 - because nfls also flags the computation
c     of the box parameters
         
c     also unset the update flag for this position

         nfls = nfls+1
         
         srcloc(nfls,1)=srcdep
         srcloc(nfls,2)=srcpos
         new(nfls)=.false.
         
         if(nfls.le.9)then
            write(ext1,'(i1)')nfls
            srcfile(nfls)='ttamp'//ext1//ftyp
         else if(nfls.le.99)then
            write(ext2,'(i2)')nfls
            srcfile(nfls)='ttamp'//ext2//ftyp
         else if(nfls.le.999)then
            write(ext3,'(i3)')nfls
            srcfile(nfls)='ttamp'//ext3//ftyp
         else if(nfls.le.9999)then
            write(ext4,'(i4)')nfls
            srcfile(nfls)='ttamp'//ext4//ftyp
         else
            write(ipdmp,*)' Error: READ_TT'
            write(ipdmp,*)' cannot deal with 10,000 files'
            ier=1
            return
         end if

c MAJOR REDESIGN of 19.08.96: the code to write out data used to be
c here - now have moved to "bump" segment as tables are written only
c when they are bumped.

      end if
            
c     cache code: store coarse grid table to cache if
c       (a) it's newly computed (sindex=0) or
c       (b) it's read from disk and not currently in cache 
c           (sindex!=0 and srcage(sindex)=0)
c     if there is room, tack it onto the end of the cache. if not,
c     bump the oldest table with the same token.
c
c     NOTE: only do this if tables are being saved, as the cache 
c     functions as a DISK CACHE!

c     first check to see whether there is enough room to implement
c     cacheing strategy at all - this requires at least room for two
c     ttamp tables. Since src and recvr tables are computed alternately,
c     this provides one of each token, so there will always be an old 
c     table of the same token to bump. If so, set the cache flag. If
c     not leave it unset - this will cause all of the table arrays to
c     remain at their default values, so the incore stuff is then never
c     activated.

c     the "bump" portion of the code attempts to overwrite an old
c     table with a new one. this will only work if all tables are 
c     the same size. to deal with variable size tables, the only
c     sensible option is to rewrite this entire routine in C/C++,
c     and use dynamic memory allocation.

      if ((save_ttflag.ne.0).and.
     &     (2*(6+ntables*n1box_s*n2box_s).le.tt_size)) then
         cache=1
         if (idbg.gt.0) then
            write(ipdmp,*)' READ_TT: cache on'
            write(ipdmp,*)' cache size        = ',tt_size
            write(ipdmp,*)' sindex            = ',sindex
            write(ipdmp,*)' top of cache      = ',top_incore
            write(ipdmp,*)' tables cached     = ',nfls_incore
         end if
      end if

      if ((cache.eq.1).and.(save_ttflag.ne.0).and.
     &     ((sindex.eq.0).or.
     &     ((sindex.ne.0).and.
     &     (srcage(sindex).eq.0)))) then

         if (idbg.gt.0) then
            write(ipdmp,*)' READ_TT: adding new table to cache'
         endif

c     set sindex if it's not

         if (sindex.eq.0) then
            sindex=nfls
         end if

c     age all existing tables

         do i=1,nfls
            if (srcage(i).gt.0) then
               srcage(i)=srcage(i)+1
            end if
         end do

c     either add table to end, or replace old table

         if (top_incore+(6+ntables*n1box_s*n2box_s).le.tt_size) then
            srcoff(sindex)=top_incore+1
            nfls_incore=nfls_incore+1
            top_incore=top_incore+ntables*n1box_s*n2box_s+6
            if (idbg.gt.0) then
               write(ipdmp,*)
     &              ' READ_TT: adding table ',sindex,' to top of cache'
               write(ipdmp,*)' new top = ',top_incore
            end if
         else

c     find the oldest table with the given token

	    if (idbg.gt.0) then
	       write(ipdmp,*)' READ_TT: bumping oldest table,',
     &               ' add new table in its place'
            endif
            oldest=0
	    oldestage=0
            do i=1,nfls
               if ((srctoken(i).eq.token).and.
     &		   (oldestage.lt.srcage(i))) then
		  oldestage=srcage(i)
                  oldest=i
                  if (idbg.gt.1) then
                     write(ipdmp,*)' READ_TT: finding oldest table',
     &                    ' to bump'
                     write(ipdmp,*)' current oldest    = ',oldest
                     write(ipdmp,*)' current oldestage = ',oldestage        
                  endif
               endif
            enddo
	    if (idbg.gt.0) then
	       write(ipdmp,*)' READ_TT: found oldest table with token',
     &               token
               write(ipdmp,*)' table index    = ',oldest
               write(ipdmp,*)' table age      = ',oldestage
            endif      
            if ((oldest.le.0).or.(oldestage.le.0)) then
               write(ipdmp,*)' Error: READ_TT'
               write(ipdmp,*)' failed to find oldest table'
               write(ipdmp,*)' something is wrong here - not enough'
               write(ipdmp,*)' cache allocated, most likely'
               ier=1
               return
            endif

c ************** NEW FILE i/o ************************

c     if save_ttflag is 1 then store records of tt/amp

	    if (save_ttflag.eq.1)then
            
               if (idbg.gt.0) then
	          write(ipdmp,*)' READ_TT:'
        	  write(ipdmp,*)
     &                 ' saving tt/amp tables for source'
	          write(ipdmp,*)
     &                 ' coordinates ',srcloc(oldest,1),', ',
     &                 srcloc(i,2)
        	  write(ipdmp,*)' in file srcfile(',oldest,') = ',
     &                 srcfile(oldest)
	       end if

c               n1box_s=nint(tt(srcoff(oldest)))
c               n2box_s=nint(tt(srcoff(oldest)+1))
c               o1box_s=tt(srcoff(oldest)+2)
c               o2box_s=tt(srcoff(oldest)+3)
c               d1box_s=tt(srcoff(oldest)+4)
c               d2box_s=tt(srcoff(oldest)+5)

c     create a C-readable string to pass as the filename

               size=index(srcfile(oldest),blank) - 1
	 
               do i=1,size
        	  charbuf(i)=srcfile(oldest)(i:i)
               end do
               charbuf(size+1)=char(0)

               call writesep(charbuf,
c     n1box_s,ntables*n2box_s,d1box_s,d2box_s,o1box_s,o2box_s,
     &              nint(tt(srcoff(oldest))),
     &              ntables*nint(tt(srcoff(oldest)+1)),
     &              tt(srcoff(oldest)+4),tt(srcoff(oldest)+5),
     &              tt(srcoff(oldest)+2),tt(srcoff(oldest)+3),
     &              tt(srcoff(oldest)+6),ntotal,idbg,ier)            
               if (ier.ne.0) then
                  write(ipdmp,*)' Error: READ_TT from WRITESEP'
                  return
               end if
               if (idbg.gt.2) then
                  write(ipdmp,*)' Return: READ_TT from WRITESEP'
                  write(ipdmp,*)' wrote ',ntotal,' floats'
               endif
            end if

c ***************** END OF NEW I/O CODE ****************************

c     having found oldest table with given token, and maybe written
c     it out to disk, bump it from cache.

c     new offset = oldest offset

            srcoff(sindex)=srcoff(oldest)

c     oldest srcage reset to zero (bumped table)

            srcage(oldest)=0

c     token variable set to negative value, indicating not in core

            srctoken(oldest)=-1

c     offset for oldest table reset to zero, indicating not in cache

            srcoff(oldest)=0

            if (idbg.gt.0) then
            write(ipdmp,*)
     &           ' READ_TT: table ',sindex,' bumping table ',oldest
            write(ipdmp,*)' from cache at offset ',
     &            srcoff(sindex)
            end if

            if (srcoff(sindex).eq.0) then
               write(ipdmp,*)' Error: READ_TT'
               write(ipdmp,*)' failed to find cache for table ',sindex
               ier=1
            end if

c ******************* END OF BUMP CODE ****************************

         end if

c     new source age = 1 (newest)

         srcage(sindex)=1

c     new source token = given token

         srctoken(sindex)=token

c     write data into cache

         tt(srcoff(sindex))=float(n1box_s)
         tt(srcoff(sindex)+1)=float(n2box_s)
         tt(srcoff(sindex)+2)=o1box_s
         tt(srcoff(sindex)+3)=o2box_s
         tt(srcoff(sindex)+4)=d1box_s
         tt(srcoff(sindex)+5)=d2box_s
         i=1
         size=ntables*n1box_s*n2box_s
         call scopy(size,work(ttamp_sptr),i,tt(srcoff(sindex)+6),i)

         if (idbg.gt.0) then
            write(ipdmp,2000)nfls,nfls_incore,tt_size,top_incore
            do i=1,nfls
               if (srcage(i).gt.0) then
                  write(ipdmp,2010)i,srcloc(i,1),srcloc(i,2),
     &                 srcoff(i),srcage(i),srctoken(i)
               end if
            end do
         end if
         if (idbg.ne.0) then
            write(ipdmp,*)' copying coarse grid ttamp table to cache'
            write(ipdmp,*)' offset     = ',srcoff(sindex)
            write(ipdmp,*)' age        = ',srcage(sindex)
            write(ipdmp,*)' n1box_s    = ',n1box_s
            write(ipdmp,*)' n2box_s    = ',n2box_s
            write(ipdmp,*)' o1box_s    = ',o1box_s
            write(ipdmp,*)' o2box_s    = ',o2box_s
            write(ipdmp,*)' d1box_s    = ',d1box_s
            write(ipdmp,*)' d2box_s    = ',d2box_s
         end if

c ***************** END OF CACHE CODE *******************************

      end if

c     interpolation: internal coarse grid ---> external grid

      call interpolate(work(ttamp_ptr),n1box,n2box,
     &     work(ttamp_sptr),n1box_s,n2box_s,ntables,tt_sam,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: READ_TT from INTERPOLATE  '
         write(ipdmp,*)' ier = ',ier
         return
      end if

      if (idbg.ge.2) then
         write(ipdmp,*)' READ_TT: dump of ttamp tables'
         write(ipdmp,*)' coarse grid traveltime:'
         do j=1,n2box_s
            write(ipdmp,*)' trace ',j
            write(ipdmp,1020)(work(ttamp_sptr-1+
     &           (j-1)*n1box_s+i),i=1,n1box_s)
         end do
         if(inversion.eq.1)then
         write(ipdmp,*)' coarse grid reciprocal of amplitudes:'
         do j=1,n2box_s
            write(ipdmp,*)' trace ',j
            write(ipdmp,1020)(work(ttamp_sptr-1+
     &           (n2box_s+j-1)*n1box_s+i),i=1,n1box_s)
         end do
          write(ipdmp,*)' coarse grid tau_x:'
          do j=1,n2box_s
            write(ipdmp,*)' trace ',j
            write(ipdmp,1020)(work(ttamp_sptr-1+
     &           (2*n2box_s+j-1)*n1box_s+i),i=1,n1box_s)
          end do
          write(ipdmp,*)' coarse grid tau_z:'
          do j=1,n2box_s
            write(ipdmp,*)' trace ',j
            write(ipdmp,1020)(work(ttamp_sptr-1+
     &           (3*n2box_s+j-1)*n1box_s+i),i=1,n1box_s)
          end do
          write(ipdmp,*)' coarse grid phi_xr:'
          do j=1,n2box_s
            write(ipdmp,*)' trace ',j
            write(ipdmp,1020)(work(ttamp_sptr-1+
     &           (4*n2box_s+j-1)*n1box_s+i),i=1,n1box_s)
          end do
          if(three_d.eq.1)then
           write(ipdmp,*)' coarse grid tau_yy:'
           do j=1,n2box_s
             write(ipdmp,*)' trace ',j
             write(ipdmp,1020)(work(ttamp_sptr-1+
     &             (5*n2box_s+j-1)*n1box_s+i),i=1,n1box_s)
           end do
	  endif
	 endif
         if(inversion.eq.0)then
            write(ipdmp,*)' coarse grid amplitudes:'
            do j=1,n2box_s
               write(ipdmp,*)' trace ',j
               write(ipdmp,1020)(work(ttamp_sptr-1+
     &              (n2box_s+j-1)*n1box_s+i),i=1,n1box_s)
            end do
            if(three_d.eq.1)then
             write(ipdmp,*)' coarse grid tau_yy:'
             do j=1,n2box_s
               write(ipdmp,*)' trace ',j
               write(ipdmp,1020)(work(ttamp_sptr-1+
     &               (2*n2box_s+j-1)*n1box_s+i),i=1,n1box_s)
             end do
	   endif
	 endif
         write(ipdmp,*)' fine grid traveltime:'
         do j=1,n2box
            write(ipdmp,*)' trace ',j
            write(ipdmp,1020)(work(ttamp_ptr-1+
     &           (j-1)*n1box+i),i=1,n1box)
         end do
         if(inversion.eq.1)then
          write(ipdmp,*)' fine grid reciprocal of amplitudes:'
          do j=1,n2box
             write(ipdmp,*)' trace ',j
             write(ipdmp,1020)(work(ttamp_ptr-1+
     &            (n2box+j-1)*n1box+i),i=1,n1box)
          end do
          write(ipdmp,*)' fine grid tau_x:'
          do j=1,n2box
            write(ipdmp,*)' trace ',j
            write(ipdmp,1020)(work(ttamp_ptr-1+
     &           (2*n2box+j-1)*n1box+i),i=1,n1box)
          end do
          write(ipdmp,*)' fine grid tau_z:'
          do j=1,n2box
            write(ipdmp,*)' trace ',j
            write(ipdmp,1020)(work(ttamp_ptr-1+
     &           (3*n2box+j-1)*n1box+i),i=1,n1box)
          end do
          write(ipdmp,*)' fine grid phi_xr:'
          do j=1,n2box
            write(ipdmp,*)' trace ',j
            write(ipdmp,1020)(work(ttamp_ptr-1+
     &           (4*n2box+j-1)*n1box+i),i=1,n1box)
          end do
	   if(three_d.eq.1)then
             write(ipdmp,*)' fine grid tau_yy:'
             do j=1,n2box
               write(ipdmp,*)' trace ',j
               write(ipdmp,1020)(work(ttamp_ptr-1+
     &              (5*n2box+j-1)*n1box+i),i=1,n1box)
             end do
	   endif
	 endif
         if(inversion.eq.0)then
           write(ipdmp,*)' fine grid amplitudes:'
           do j=1,n2box
              write(ipdmp,*)' trace ',j
              write(ipdmp,1020)(work(ttamp_ptr-1+
     &             (n2box+j-1)*n1box+i),i=1,n1box)
          end do
          if(three_d.eq.1)then
             write(ipdmp,*)' fine grid tau_yy:'
             do j=1,n2box
               write(ipdmp,*)' trace ',j
               write(ipdmp,1020)(work(ttamp_ptr-1+
     &              (2*n2box+j-1)*n1box+i),i=1,n1box)
             end do
	  endif
       end if
      end if
         
 1000 format(i10)
 1010 format(e10.4,2x,e10.4,2x,a20,2x,i10,2x,i10,2x,i10)
 1020 format(6(e10.4,2x))
 2000    format(//,
     &     ' READ_TT: CACHE STATUS',/,
     &     ' number of tables computed     = ',i3,/,
     &     ' number of tables cached       = ',i3,/,
     &     ' size of cache                 = ',i10,/,
     &     ' top of filled segment         = ',i10,//,
     &        ' index    z            x      offset    age  token')
 2010    format(i3,2x,e10.4,2x,e10.4,2x,i8,2x,i3,4x,i3)

      if (idbg.gt.0) then
         write(ipdmp,*)' Leaving READ_TTDATA'
         write(ipdmp,*)' workspace available = ',lenwork
      endif

      return 
      end

c====================================================================

       subroutine interpolate (ddfine, n1fine, n2fine,
     &                         ddgros, n1gros, n2gros,ntables,
     &                         nratio, ipdmp, ier)
c----------------------------------------------------------
c      This subroutine interpolates array "ddfine" onto
c      array "ddgros". NOTE that the arrays are three-
c      dimensional. We have to interpolate for all the
c      3 dimensions. Currently we allow a ratio equal to
c      1, 2 and 4 for nratio.
c
c      Roelof VERSTEEG, 07/93
c      Huy TRAN,        07/93
c----------------------------------------------------------
	   integer n1gros, n2gros, n1fine, n2fine
	   integer nright, nlower
	   integer nratio, ipdmp, ier
	   integer i, j, k
	   integer ntables

	   real ddgros(n1gros, n2gros, ntables)
	   real ddfine(n1fine, n2fine, ntables)
	   real hal1, qua1, qua3, eig1 
	   real six1, six3, six9
	   real dhupp,dhlow,dvlef
	   real dvrig,diag2,diag1

	   logical badratio, baddimen
c==========================================================

c--  check ratio
c-----------------
         badratio= (nratio.ne.1).and.(nratio.ne.2)
     &                          .and.(nratio.ne.4)
         if (badratio) then
	   write(ipdmp,*) ' ttime grid sampling factor : ', nratio
	   write(ipdmp,*) ' Sorry, this must be either 1, 2 or 4 '
	   ier= 712
	   return
	 end if

c--  check array dimensions
c----------------------------
	 nright= 1+ (n1gros-1)* nratio
	 nlower= 1+ (n2gros-1)* nratio
	 baddimen= ( (nright.gt.n1fine).or.(nlower.gt.n2fine) )
         if (baddimen) then
	   write(ipdmp,*) ' Sorry, bad dimension specifications '
	   ier= 713
	   return
	 end if

c--  case nratio= 1
c--------------------
         if (nratio.eq.1) then

	   do 10 k=1, ntables
	     do 11 j=1, n2gros
	     do 12 i=1, n1gros
	       ddfine(i,j,k)= ddgros(i,j,k)
 12          continue
 11          continue
 10        continue

         end if

c--  case nratio= 2
c--------------------
	 if (nratio.eq.2) then
	   
	   hal1= 0.50
	   qua1= 0.25

	   do 20 k= 1, ntables

c--    interior points
	     do 21 j= 1, n2gros-1  
	     do 22 i= 1, n1gros-1
	       ddfine(1+(i-1)*2, 1+(j-1)*2, k)
     &              = ddgros(i,j,k)
               ddfine(2*i, 1+(j-1)*2, k)
     &              = hal1* (ddgros(i,j,k)+ ddgros(i+1,j,k))
               ddfine(1+(i-1)*2, 2*j, k)
     &              = hal1* (ddgros(i,j,k)+ ddgros(i,j+1,k))
	       ddfine(2*i, 2*j, k)
     &              = qua1* (ddgros(i,j,k)  + ddgros(i+1,j,k)
     &                     + ddgros(i,j+1,k)+ ddgros(i+1,j+1,k))
 22          continue
 21          continue

c--    lower-boundary points
             do 23 i= 1, n1gros-1
	       ddfine(1+(i-1)*2, nlower, k)
     &              = ddgros(i,n2gros,k)
	       ddfine(2*i, nlower, k)
     &              = hal1* ( ddgros(i,n2gros,k)
     &                      + ddgros(i+1,n2gros,k) )
 23          continue

c--    right-boundary points
             do 24 j= 1, n2gros-1
	       ddfine(nright, 1+(j-1)*2, k)
     &              = ddgros(n1gros,j,k)
	       ddfine(nright, 2*j, k)
     &              = hal1* ( ddgros(n1gros,j,k)
     &                      + ddgros(n1gros,j+1,k) )
 24          continue

c--    lower-right point
	     ddfine(nright, nlower, k)= ddgros(n1gros,n2gros,k)

 20        continue

         end if

c--  case nratio= 4
c--------------------
         if (nratio.eq.4) then

           hal1= 1./2.
           qua1= 1./4. 
	   qua3= 3./4. 
	   eig1= 1./8.
	   six1= 1./16. 
	   six3= 3./16. 
	   six9= 9./16. 

	   do 40 k= 1, ntables
c	     do 41 j= 1, n2gros-3 
	     do 41 j= 1, n2gros-1
c	     do 42 i= 1, n1gros-3
	     do 42 i= 1, n1gros-1
	       ddfine(1+(i-1)*4, 1+(j-1)*4, k)
     &              = ddgros(i,j,k) 
               ddfine(2+(i-1)*4, 1+(j-1)*4, k)
     &              = qua3* ddgros(i,j,k)+ qua1* ddgros(i+1,j,k)
	       ddfine(4*i, 1+(j-1)*4, k)
     &              = qua1* ddgros(i,j,k)+ qua3* ddgros(i+1,j,k)
               ddfine(1+(i-1)*4, 2+(j-1)*4, k)
     &              = qua3* ddgros(i,j,k)+ qua1* ddgros(i,j+1,k)
               ddfine(1+(i-1)*4, 4*j, k)
     &              = qua1* ddgros(i,j,k)+ qua3* ddgros(i,j+1,k)

c--    here the computations are arranged so as to make
c      triadic sums, i.e. sums of the form X + bY, where
c      X and Y are vectors, "b" a scalar, appear in an
c      easily compiler-recognizable way.
c---------------------------------------------------------
	       dhupp= eig1* (ddgros(i,j,k)  + ddgros(i+1,j,k))
	       dhlow= eig1* (ddgros(i,j+1,k)+ ddgros(i+1,j+1,k))
               ddfine(3+(i-1)*4, 1+(j-1)*4, k)
     &              = 4.* dhupp 
	       ddfine(3+(i-1)*4, 2+(j-1)*4, k)
     &              = 3.* dhupp + dhlow
	       ddfine(3+(i-1)*4, 4*j, k)
     &              = 3.* dhlow + dhupp
               ddfine(3+(i-1)*4, 3+(j-1)*4, k)
     &              = 2.* (dhupp + dhlow)

	       dvlef= eig1* (ddgros(i,j,k)  + ddgros(i,j+1,k))
	       dvrig= eig1* (ddgros(i+1,j,k)+ ddgros(i+1,j+1,k))
	       ddfine(1+(i-1)*4, 3+(j-1)*4, k)
     &              = 4.* dvlef 
	       ddfine(2+(i-1)*4, 3+(j-1)*4, k)
     &              = 3.* dvlef + dvrig 
	       ddfine(4*i, 3+(j-1)*4, k)
     &              = 3.* dvrig+ dvlef 

	       diag2= six3* (ddgros(i+1,j,k)+ ddgros(i,j+1,k))
               ddfine(2+(i-1)*4, 2+(j-1)*4, k)
     &              = six9* ddgros(i,j,k)+ six1* ddgros(i+1,j+1,k)
     &                    + diag2 
	       ddfine(4*i, 4*j, k)
     &              = six1* ddgros(i,j,k)+ six9* ddgros(i+1,j+1,k)
     &                    + diag2 

               diag1= six3* (ddgros(i,j,k)+ ddgros(i+1,j+1,k))
               ddfine(4*i, 2+(j-1)*4, k)
     &              = six9* ddgros(i+1,j,k)+ six1* ddgros(i,j+1,k)
     &                    + diag1 
               ddfine(2+(i-1)*4, 4*j, k)
     &              = six1* ddgros(i+1,j,k)+ six9* ddgros(i,j+1,k)
     &                    + diag1 

 42          continue
 41          continue

c--    lower-boundary points
c             do 43 i= 1, n1gros-3
             do 43 i= 1, n1gros-1
	       ddfine(1+(i-1)*4, nlower, k)
     &              = ddgros(i,n2gros,k)
               ddfine(2+(i-1)*4, nlower, k)
     &              = qua3* ddgros(i,n2gros,k)
     &              + qua1* ddgros(i+1,n2gros,k)
	       ddfine(4*i, nlower, k)
     &              = qua1* ddgros(i,n2gros,k)
     &              + qua3* ddgros(i+1,j,k)
               ddfine(3+(i-1)*4, 1+(j-1)*4, k)
     &              = hal1* ( ddgros(i,n2gros,k)
     &                      + ddgros(i+1,n2gros,k) )
 43          continue

c--    right-boundary points
c             do 44 j= 1, n2gros-3
             do 44 j= 1, n2gros-1
	       ddfine(nright, 1+(j-1)*4, k)
     &              = ddgros(n1gros,j,k)
               ddfine(nright, 2+(j-1)*4, k)
     &              = qua3* ddgros(n1gros,j,k)
     &              + qua1* ddgros(n1gros,j+1,k)
	       ddfine(nright, 4*j, k)
     &              = qua1* ddgros(n1gros,j,k)
     &              + qua3* ddgros(n1gros,j+1,k)
               ddfine(nright, 3+(j-1)*4, k)
     &              = hal1* ( ddgros(n1gros,j,k)
     &                      + ddgros(n1gros,j+1,k) )
 44          continue

c--    lower-right corner
             ddfine(nright, nlower, k)= ddgros(n1gros, n2gros,k)

 40        continue
	 end if

      return
      end

c=============================================================================
