       
c******************************************************************

      subroutine eikonal_rhs(nx, nz, iz, ss, tt, ham, tx, txx, apcut,
     &     dx, dztmp, dz, step_total, adjust, lenwork, 
     &     eps, work, idbg, ipdmp, ier)

c---------------------------------------------------------------
c      Computes a numerical approximation to the Hamiltonian
c      by means of a 2nd order upwind finite-difference scheme
c      For details see paper by Osher & Sethian (1988).
c
c      MATLAB VERSION:
c
c function [tx,txx,h,g]=hamu(ss,tt,dx,dz,eikeps,ier);
c [nz,nx]=size(ss);
c zz=zeros(1,nx);
c 
c cfl=0.4;
c 
c % divided differences to order 3
c 
c diff=zeros(3,nx);
c diff(1,1:nx-1)= (tt(2:nx)-tt(1:(nx-1)))/dx;
c diff(2,1:nx-2)= (diff(1,2:nx-1)-diff(1,1:nx-2))/dx;
c 
c % 2nd order corrections
c 
c cm=[0, 0, diff(2,1:nx-2)];
c cp=[0, diff(2,1:nx-2), 0];
c txxm=mso(cm,cp);
c dma=[0, diff(1,1:nx-1)] + 0.5*dx*txxm;
c cm=[0, diff(2,1:nx-2), 0];
c cp=[diff(2,1:nx-2), 0, 0];
c txxp=mso(cm,cp);
c dpa=[diff(1,1:nx-1), 0] - 0.5*dx*txxp;
c 
c % first and second differences to be passed to transport solver
c
c tx=max(dma,zz)+min(dpa,zz);
c txx=cm;
c txx(1)=txx(2);
c txx(nx)=txx(nx-1);
c
c % compute hamiltonian and cutoff function
c 
c a=(max(dma,zz)).^2 + (min(dpa,zz)).^2;
c [b,g]=blunt(max(ss-a,zz),eikeps*eikeps*ss);
c h=sqrt(b);
c 
c % check cfl condn
c 
c ier=0;
c cfltest = dz*a/(dx*h);
c if cfltest > cfl
c    fprintf(' Error: HAM3\n');
c    fprintf(' CFL condition violated\n');
c    fprintf(' c = %g cfltest = %g cfl = %g\n',a/h,cfltest,cfl);
c    ier=1;
c    error('Returning to TTAMP')
c end
c 
c 
c function [h,g]=blunt(x,xcut);
c 
c tol=1.0e-10;
c h=min(0.5*(xcut+x.*x./max(tol*ones(x),xcut)),max(x,xcut));
c g=herm(x,0.5*xcut,0.0,xcut,1.0);
c 
c function zo=herm(x,a,xa,b,xb);
c
c tol=1.0e-10;
c r=min(b-a,max(zeros(x),x-a));
c k=max(tol*ones(x),b-a);
c r=r./k;
c zo=(xb-xa)*(-2*r.^3 + 3*r.^2) + xa*ones(x);
c 
c---------------------------------------------------------------

c-- Arguments

      integer
     &     nx,nz,               ! grid dimensions
     &     iz,                  ! current depth step
     &     lenwork,             ! length of "work"
     &     idbg, ipdmp, ier     ! debug and error

      real 
     &     eps,                 ! eikonal threshhold
     &     step_total,          ! total step taken so far
     &     dx,                  ! horizontal step
     &     dz,                  ! vertical step
     &     dztmp                ! vertical step - full step on call

      real ss(nz,nx),           ! square slowness
     &     tt(nx),              ! travel time at current depth
     &     ham(nx),             ! numerical hamiltonian - rhs for eik. scheme
     &     tx(nx),              ! ENO first difference of tt
     &     txx(nx+2),           ! centered 2nd difference of tt
     &     apcut(nx),           ! Hermite tapered cutoff
     &     work(*)              ! workspace

c-- Internal Variables

      real
     &     dtt2, cfl, tol, zero, compa, relstep, dxdiv,
     &     sqeps

      real tmp1, tmp2, tmp3, tmp4, half

      integer
     &     nxm1, j,
     &     dppptr, dpmptr, dmmptr, dpaptr, dmaptr,
     &     ssptr, workptr

      logical adjust

      data cfl /0.30/, tol /1.e-8/

c==================================================================

      if (ier.ne.0) return

c-------------------------
c--  Debug instructions
c-------------------------
      if (idbg.ge.1) then
         write(ipdmp,*) ' EIK_RHS'
      end if
      if (idbg.ge.2) then
         write(ipdmp,*)' inputs:'
         write(ipdmp,*) ' nx               = ', nx
         write(ipdmp,*) ' dx               = ', dx
         write(ipdmp,*) ' dz               = ', dz
         write(ipdmp,*) ' dztmp            = ', dztmp
         write(ipdmp,*) ' step in Hamilton = ', step_total
         write(ipdmp,*) ' adjust           = ', adjust
         write(ipdmp,*) ' eps              = ', eps
         write(ipdmp,*) ' ss: '
         write(ipdmp,1000)(ss(iz,j),j=1,nx)
         write(ipdmp,*) ' tt: '
         write(ipdmp,1000)(tt(j),j=1,nx)
      end if

c--  Useful numbers
c---------------------
      zero= 0.0e+00

c set eikonal threshhold - note this is SQUARE of cosine of max 
c incidence angle

      sqeps = eps**2

c--  Allocate memory
c----------------------
      workptr= 1
      call getbuf('work', ssptr, nx, workptr, lenwork,
     &     ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*) ' Error: EIK_RHS from GETBUF for ssptr'
         return
      end if

c interpolate square slowness field

      relstep = (dz - step_total)/dz

      dxdiv=1/dx
      do j=1,nx-1

         work(ssptr-1+j)=relstep*ss(iz,j) + 
     &        (1.0e+00 - relstep)*ss(iz+1,j)
         tx(j)= (tt(j+1)- tt(j))*dxdiv

      end do

      work(ssptr-1+nx)=relstep*ss(iz,nx) + 
     &     (1.0e+00 - relstep)*ss(iz+1,nx)
      tx(nx)=tx(nx-1)

c dump inputs and slopes

      if (idbg.ge.2) then
         write(ipdmp,*) ' input tt:'
         write(ipdmp,1000) (tt(j), j= 1, nx)
         write(ipdmp,*) ' input ss (interpolated):'
         write(ipdmp,1000) (work(ssptr-1+j), j= 1, nx)
         write(ipdmp,*) ' slope (1 div diff)'
         write(ipdmp,1000) (tx(j),j=1,nx)
      end if

c--  Check for inflow at ends, error return if detected
c---------------------------------------------------------

      if (tx(1).gt.zero) then
         write(ipdmp,*) ' Error: EIKONAL_RHS'
         write(ipdmp,*) ' inflow at left endpoint'
         write(ipdmp,*) ' dxt = ', tx(1)
         ier= 1
         write(ipdmp,*) ' input tt:'
         write(ipdmp,1000) (tt(j), j= 1, nx)
         write(ipdmp,*) ' input ss (interpolated):'
         write(ipdmp,1000) (work(ssptr-1+j), j= 1, nx)
         write(ipdmp,*) ' slope (1 div diff)'
         write(ipdmp,1000) (tx(j),j=1,nx)
         return
      end if

      if (tx(nx-1).lt.zero) then
         write(ipdmp,*) ' Error: EIKONAL_RHS'
         write(ipdmp,*) ' inflow at right endpoint'
         write(ipdmp,*) ' dxt = ', tx(nx-1)
         ier= 1
         write(ipdmp,*) ' input tt:'
         write(ipdmp,1000) (tt(j), j= 1, nx)
         write(ipdmp,*) ' input ss (interpolated):'
         write(ipdmp,1000) (work(ssptr-1+j), j= 1, nx)
         write(ipdmp,*) ' dxtptr '
         write(ipdmp,1000) (tx(j),j=1,nx)
         return
      end if

c Form second difference of tt

      txx(1)  = zero
      txx(2)  = zero
      do j= 1,nx-2
         txx(2+j)=(tx(j+1)-tx(j))*dxdiv
      end do
      txx(nx+1)= zero
      txx(nx+2)= zero

c define pointers to retarded, centered, and advanced second diffs

      dmmptr= 1
      dpmptr= 2
      dppptr= 3
      
c ENO corrections

      call getbuf('work', dpaptr, nx, workptr, lenwork,
     &     ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*) ' Error: EIK_RHS from GETBUF for dpaptr'
         return
      end if
      
      call getbuf('work', dmaptr, nx, workptr, lenwork,
     &     ipdmp, ier)
      if (ier.ne.0) then
         write(ipdmp,*) ' Error: EIK_RHS from GETBUF for dmaptr'
         return
      end if

      nxm1= nx- 1
      call mso(nxm1, txx(dmmptr), txx(dpmptr),
     &     work(dmaptr+1))
      if (ier.ne.0) then
         write(ipdmp,*) ' Error: EIK_RHS from MSO'
         return
      end if
      
      call mso(nxm1, txx(dpmptr), txx(dppptr),
     &     work(dpaptr))
      if (ier.ne.0) then
         write(ipdmp,*) ' Error: EIK_RHS from MSO'
         return
      end if

c--  The computations of "dma" and "dpa" have been
c    gathered in one single loop for vector "dttx"
c    to stay in the register.
c----------------------------------------------------

CDIR$ NORECURRENCE
      do j= 1, nx-1
         work(dmaptr+j)  = tx(j)
     &        + 0.5*dx*work(dmaptr+j)
         work(dpaptr-1+j)= tx(j)
     &        - 0.5*dx*work(dpaptr-1+j)
      end do
      work(dmaptr)     = zero
      work(dpaptr-1+nx)= zero

c--  Compute the Hamiltonian; interleave computation of CFL step,
c    but only change dztmp if desired
c    THRESHHOLD at sqeps*ss, quadratic taper to 0.5*sqeps*ss;
c    also return Hermite cutoff
c
c VERSION WITH NO IFs:
c
      half=0.5e+00
      compa = dz - step_total
      do j= 1, nx
         tmp3=max(work(dmaptr-1+j),zero)
         tmp4=min(work(dpaptr-1+j),zero)
         tx(j)=tmp3 + tmp4
         dtt2=tmp3**2 + tmp4**2
         tmp1=work(ssptr-1+j)-dtt2
         tmp2=sqeps*work(ssptr-1+j)
         ham(j)=sqrt(max(tmp1,tmp2))
         apcut(j)=0.5e+00+sign(half,tmp1-tmp2)
c
c VERSION WITH IFs:
c
c         dtt2=
c     &        (max(work(dmaptr-1+j), zero))**2 +
c     &        (min(work(dpaptr-1+j), zero))**2 
c         if (dtt2.lt.(1-sqeps)*work(ssptr-1+j)) then
c            ham(j)=sqrt(work(ssptr-1+j)-dtt2)
c            apcut(j)=1.0e+00
c            tx(j)=
c     &           max(work(dmaptr-1+j), zero) +
c     &           min(work(dpaptr-1+j), zero)
c         else
c            ham(j)=sqeps*work(ssptr-1+j)
c            apcut(j)=0.0e+00
c            if (work(dmaptr-1+j).gt.zero) then
c               tx(j)=sqrt(work(ssptr-1+j)-ham(j))
c            else
c               tx(j)=-sqrt(work(ssptr-1+j)-ham(j))
c            end if
c            ham(j)=sqrt(ham(j))
c         endif
         dtt2=sqrt(abs(dtt2))
         compa= min(compa, cfl* dx* ham(j)/(max(dtt2,tol)))
      end do

      if (idbg.ge.2) then
         write(ipdmp,*) ' dma '
         write(ipdmp,1000) (work(dmaptr-1+j),j=1,nx)
         write(ipdmp,*) ' dpa '
         write(ipdmp,1000) (work(dpaptr-1+j),j=1,nx)
         write(ipdmp,*) ' dxt '
         write(ipdmp,1000) (tx(j),j=1,nx)
         write(ipdmp,*) ' dxxt '
         write(ipdmp,1000) (txx(j),j=1,nx)
         write(ipdmp,*) ' hamiltonian:'
         write(ipdmp,1000) (ham(j), j= 1, nx)
         write(ipdmp,*) ' aperture:'
         write(ipdmp,1000) (apcut(j),j=1,nx)
      end if

c--  Safeguarded CFL step selection
c-------------------------------------

      if (adjust) then
         dztmp= min(dz-step_total, compa)
         if (idbg.ge.1) then
            write(ipdmp,*) ' dztmp recommended = ', dztmp
         end if
      end if

 1000 format(6(e10.4,2x))

      return
      end
