c=======================BELK_DET=========================================
c This subroutine calculates the mixed (x,x_r) derivative of the
c traveltime that arises as the result of the evaluation of the
c Beylkin Determinant.
c
      subroutine belk_det(nx,nz,dx,dz,a,u_x,tau_d,
     &     alpha,work,len_work,ipdmp,ier)
c
c
c===========================================================
c
c     This subroutine computes the solution of 
c
c  	| u_z + a u_x = 0.0
c  	|
c  	| u(x,z=zm) = src = tau_d
c
c     using the scheme in subroutine solve_local
c
c===========================================================
      
c---
c Input arguments 
c---
      integer 
     &     nx,nz,             ! dimension of the array u
     &     work_ptr,          ! first available workspace
     &     len_work,          ! length of available workspace
     &     ipdmp,ier          ! dump unit, error flag

      integer
     &     alpha             ! step variable
      real
     &     dx,dz,             ! x and z steps
     &     a(nz,nx),          ! input coefficient field
     &     tau_d(nx),         ! partial(tau_r)/partial(x_r) at the datum
     &     work(*)            ! workspace
c---
c Output arguments 
c---
      real 
     &     u_x(nz,nx)        ! partial^2 (tau)/partial x partial x_r

c---
c Local Variables
c---
      integer
     &     i,j,k,               ! index
     &     a_int_ptr,           ! pointer to the interpolated a
     &     az_ptr,              ! pointer to the original az (z derivative a)
     &     az_int_ptr,          ! pointer to the interpolated az
     &     v_int_ptr,           ! pointer to the interpolated v
     &     a_ptr,               ! pointer to a
     &     az1_ptr,             ! pointer to az1
     &     u_x_ptr,             ! pointer to u_x
     &     u_ptr,               ! pointer to u
     &     v_ptr,               ! pointer to v
     &     idbg,                ! debug level
     &     size                 ! guess
      
      real
     &     dy                   ! new step


      if (ier.ne.0) return

c      call get_debug_level('belk_det','idbg',idbg,ier)

      if(idbg.gt.0) then
         write(ipdmp,*)' ---> BELK_DET:'
         write(ipdmp,*)' nx       =',nx
         write(ipdmp,*)' nz       =',nz
         write(ipdmp,*)' dx       =',dx
         write(ipdmp,*)' dz       =',dz
         write(ipdmp,*)' alpha    =',alpha
         write(ipdmp,*)' len_work =',len_work
      endif

      if(idbg.ge.2) then
         do j=1,nz
            write(ipdmp,837)j,(a(j,i),i=1,nx)
         enddo
      endif

837   format(' BELK_DET: a trace ',i5,/,6(e10.4,2x))

c---
c Taper of the perturbation on the boundary of the domain
c---  
      if( (nx.lt.10).or.(nz.lt.10) ) then
         write(ipdmp,*)' Domain too small '
         write(ipdmp,*)' Can''t taper the source  of the '
         write(ipdmp,*)' perturbation of the eikonal equation'
         ier=199
         return
      endif
c---
c Get space for u array: 
c---
c
      work_ptr = 1
      call getbuf('work',u_ptr,nx,work_ptr,
     &     len_work,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: BELK_DET from GETBUF: 
     &        U_PTR '
         return
      end if     

c--
c Boundary condition: The initial data at the surface:
c--

      do i=1,nx
         work(u_ptr+i-1)=tau_d(i)
      enddo

c---
c Get space for intermidiate arrays 
c---
      call getbuf('work',a_ptr,nx,work_ptr,
     &     len_work,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: BELK_DET from GETBUF: 
     &        A_PTR '
         return
      end if     
      
      call getbuf('work',az1_ptr,nx,work_ptr,
     &     len_work,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: BELK_DET from GETBUF: 
     &        AZ1_PTR '
         return
      end if     

      call getbuf('work',u_x_ptr,nx,work_ptr,
     &     len_work,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: BELK_DET from GETBUF: 
     &        U_x_PTR '
         return
      end if     
      
      call getbuf('work',v_ptr,nx,work_ptr,
     &     len_work,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: BELK_DET from GETBUF: 
     &        V_PTR '
         return
      end if     

      call getbuf('work',az_ptr,nx*nz,work_ptr,
     &     len_work,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: BELK_DET from GETBUF: 
     &        AZ_PTR '
         return
      end if     

      call der_cen_z(work(az_ptr),a,nx,nz,dz,ipdmp,ier)
c
      if (alpha.gt.1) then
         
         size=(alpha+1)*nx
c---
c Get space for interpolation  arrays: 
c---
            call getbuf('work',v_int_ptr,size,work_ptr,
     &           len_work,ipdmp,ier)
            if (ier.ne.0) then
               write(ipdmp,*)' Error: BELK_DET from GETBUF: V_INT_PTR '
               return
            end if     

            call getbuf('work',a_int_ptr,size,work_ptr,
     &           len_work,ipdmp,ier)
            if (ier.ne.0) then
               write(ipdmp,*)' Error: BELK_DET from GETBUF: 
     &          A_INT_PTR '
               return
            end if     

            call getbuf('work',az_int_ptr,size,work_ptr,
     &           len_work,ipdmp,ier)
            if (ier.ne.0) then
               write(ipdmp,*)' Error: BELK_DET from GETBUF: 
     &          AZ_INT_PTR '
               return
            end if     

c---
c Next proceed to do linear interpolation of the coefficients
c---
         do j=1,nz-1

            do i=1,nx
               work(a_ptr+i-1)=a(j,i)
               work(az1_ptr+i-1)=a(j+1,i)
            enddo

            call interp_lin(work(a_int_ptr),work(a_ptr),
     &           work(az1_ptr),nx,alpha,ipdmp,ier)

            do i=1,nx
               work(a_ptr+i-1)=work(az_ptr+(j-1)*nx+i-1)
               work(az1_ptr+i-1)=work(az_ptr+j*nx+i-1)
            enddo
c
            call interp_lin(work(az_int_ptr),work(a_ptr),
     &           work(az1_ptr),nx,alpha,ipdmp,ier)

            dy=dz/alpha
c
            do i=1,nx
               work(v_int_ptr+i-1)=work(u_ptr+i-1)
            enddo
c
            do i=2,alpha+1                            
               call solve_local(nx,nz,dx,dy,work(a_int_ptr+(i-1)*nx),
     &         work(az_int_ptr+(i-1)*nx),work(u_x_ptr),
     &         work(v_int_ptr+(i-1)*nx),
     &         work(v_int_ptr+(i-2)*nx),
     &         work(work_ptr),len_work,ipdmp,ier)
c
               if(i.eq.2)then
                  do k=1,nx
                     u_x(j,k)=work(u_x_ptr+k-1)
                  enddo
	          endif
            enddo
c
            if (ier.ne.0) then
               write(ipdmp,*)' Error: BELK_DET from SOLVE_LOCAL'
               return
            end if
c
            do i=1,nx
               work(u_ptr+i-1)=work(v_int_ptr+(alpha)*nx+i-1)
            enddo
c     
         enddo
      else
         do j=1,nz-1
            do i=1,nx
               work(a_ptr+i-1)  =a(j,i)
               work(az1_ptr+i-1)=work(az_ptr+(j-1)*nx+i-1)
               work(v_ptr+i-1)=work(u_ptr+i-1)
            enddo
c
            if (idbg.ge.1) then
              write(ipdmp,*)' BELK_DET ---> SOLVE_LOCAL'
            end if
c
            call solve_local(nx,nz,dx,dz,work(a_ptr),work(az1_ptr),
     &           work(u_x_ptr),work(u_ptr),
     &           work(v_ptr),work(work_ptr),len_work,ipdmp,ier)
c
            if (ier.ne.0) then
               write(ipdmp,*)' Error: BELK_DET from SOLVE_LOCAL'
               return
            end if
c

            do i=1,nx
               u_x(j,i)=work(u_x_ptr+i-1)
            enddo

         enddo         

      endif

      call der_cen_x(work(u_x_ptr),work(u_ptr),nx,dx,ipdmp,ier)
      
      do i=1,nx
         u_x(j,i)=work(u_x_ptr+i-1)
      enddo
c

 1040 format(6(e10.4,2x))

      return 
      end 
c===========================================================
      subroutine solve_local(nx,nz,dx,dz,a,az,u_x,u,v,
     &     work,len_work,ipdmp,ier)
c===========================================================
c
c     This subroutine computes the one step of the scheme
c     to solve the perturbation equation of the eikonal:
c
c  	| u_z + a u_x = 0.0
c  	|
c  	| u(x,z=zm) = src
c
c     So the subroutine iterates the second order scheme 
c     given by:
c
c     u^n+1 = u^n + dz.rhs_x + 0.5.dz.dz.rhs_z1
c 
c     with 
c     
c     rhs_x  = -a.d0(u)
c     rhs_z1 = a^2.u_xx + (a.d0(a) - a_z).u_x
c     
c     On the boundary we have a first order upwind scheme:
c
c     u^n+1 = u^n - am(1).dz/dx(u(2)-u(1))
c     u^n+1 = u^n - ap(nx).dz/dx(u(nx)-u(nx-1))
c
c===========================================================
      
c---
c Arguments 
c---
      integer 
     &     nx,nz,            ! dimension of the array u
     &     work_ptr,         ! first available workspace
     &     len_work,         ! length of available workspace
     &     ipdmp,ier         ! dump unit, error flag

      real
     &     dx,dz,            ! x and z steps
     &     a(nx),            ! input coefficient field
     &     az(nx),           ! its derivative in z 
     &     u_x(nx),          ! output field (partial^2 (tau)/partial x partial x_r)
     &     u(nx),            ! output field (level n+1)
     &     v(nx),            ! input field (level n)
     &     work(*)           ! workspace

c---
c Local Variables
c---
      integer
     &     i,                   ! index, debug flag 
     &     min_a_ptr,           ! pointer to the array am=min(a,0)
     &     max_a_ptr,           ! pointer to the array ap=max(a,0)
     &     rhs_x_ptr,           ! pointer to the array rhs_x
     &     rhs_z1_ptr,          ! pointer to the array rhs_z1
     &     rhs_z2_ptr,          ! pointer to the array rhs_z2
     &     a_x_ptr,             ! pointer to the array a_x
     &     u_x_ptr              ! pointer to the array u_x

      real tol                  ! tolerance for inflow test

c===========================================================================

      data tol /1.0e-05/
c
      if (ier.ne.0) return

c      
c     get some workspace
      work_ptr = 1
      call getbuf('work',min_a_ptr,nx,work_ptr,
     &     len_work,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: SOLVE_LOCAL from GETBUF: AM_PTR '
         return
      end if

      call getbuf('work',max_a_ptr,nx,work_ptr,
     &     len_work,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: SOLVE_LOCAL from GETBUF: AP_PTR '
         return
      end if

      call getbuf('work',rhs_x_ptr,nx,work_ptr,
     &     len_work,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: SOLVE_LOCAL from GETBUF: RHS_X_PTR '
         return
      end if

      call getbuf('work',rhs_z1_ptr,nx,work_ptr,
     &     len_work,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: SOLVE_LOCAL from GETBUF: RHS_Z1_PTR '
         return
      end if

      call getbuf('work',rhs_z2_ptr,nx,work_ptr,
     &     len_work,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: SOLVE_LOCAL from GETBUF: RHS_Z2_PTR '
         return
      end if

      call getbuf('work',u_x_ptr,nx,work_ptr,
     &     len_work,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: SOLVE_LOCAL from GETBUF: U_X_PTR '
         return
      end if

      call getbuf('work',a_x_ptr,nx,work_ptr,
     &     len_work,ipdmp,ier)
      if (ier.ne.0) then
         write(ipdmp,*)' Error: SOLVE_LOCAL from GETBUF: A_X_PTR '
         return
      end if

      do i=1,nx
         work(min_a_ptr+i-1)=min(a(i),0.0e+00)
         work(max_a_ptr+i-1)=max(a(i),0.0e+00)
      enddo
c    
      if( (a(nx-2).lt.tol)
     &     .or.(a(nx-1).lt.tol)
     &     .or.(a(nx).lt.tol)
     &     .or.(a(3).gt.-tol)
     &     .or.(a(2).gt.-tol)
     &     .or.(a(1).gt.-tol )) then 

         write(ipdmp,*)' ERROR SOLVE_LOCAL:'
         write(ipdmp,*)' Inflow at the boundary in the solution '
         write(ipdmp,*)' of the linearized eikonal equation'
         write(ipdmp,*)' am(nx-2) =',work(min_a_ptr+nx-3)
         write(ipdmp,*)' am(nx-1) =',work(min_a_ptr+nx-2)
         write(ipdmp,*)' am(nx)   =',work(min_a_ptr+nx-1)
         write(ipdmp,*)' ap(1)    =',work(max_a_ptr)
         write(ipdmp,*)' ap(2)    =',work(max_a_ptr+1)
         write(ipdmp,*)' ap(3)    =',work(max_a_ptr+2)
         ier=98
         return
      endif

c---
c Computation of the rhs multiplied by dz: - a.u_x
c---

      call der_cen_x(work(u_x_ptr),v,nx,dx,ipdmp,ier)
c
      do i=1,nx
	 u_x(i) = work(u_x_ptr+i-1)
         work(rhs_x_ptr+i-1)=-a(i)*work(u_x_ptr+i-1)
      enddo

c---
c a.a.u_xx
c---

      call laplacian(work(u_x_ptr),v,nx,dx,ipdmp,ier)

      do i=1,nx
         work(rhs_z1_ptr+i-1)=a(i)*a(i)*work(u_x_ptr+i-1) 
      enddo
      
c---
c (a.a_x - a_z).u_x
c---

      call der_cen_x(work(a_x_ptr),a,nx,dx,ipdmp,ier)

      do i=1,nx
         work(rhs_z1_ptr+i-1)=work(rhs_z1_ptr+i-1) + 
     &    (a(i)*work(a_x_ptr+i-1) - az(i))*work(u_x_ptr+i-1)
      enddo

      do i=2,nx-1
         u(i)=v(i) + dz*work(rhs_x_ptr+i-1) + 
     &        0.5*dz*dz*work(rhs_z1_ptr+i-1)
      enddo

      u(1)=v(1)-work(min_a_ptr)*dz/dx*(v(2)-v(1))
      u(nx)=v(nx)-work(max_a_ptr+nx-1)*dz/dx*(v(nx)-v(nx-1))

      return 
      end 


c=====================================================================
c
c
c
      subroutine der_cen_x(u_x,u,nx,dx,ipdmp,ier)

c     Arguments

      integer 
     &     nx,               ! dimension of the array u
     &     ipdmp,ier         ! dump unit, error flag

      real
     &     dx,               ! x step
     &     u(nx),            ! input vector 
     &     u_x(nx)            ! output vector (x derivative)

c     Internal Variables

      integer 
     &     i                 ! index of array 

c--------------------------------------------------------
c      write(6,*)' I am inside der_cen_x'

      do i=2,nx-1
         u_x(i)=(u(i+1)-u(i-1))/(2*dx)
      end do
      u_x(1) = (u(3)-4*u(2)+3*u(1))/(-2*dx)
      u_x(nx) = (u(nx-2)-4*u(nx-1)+3*u(nx))/(2*dx)

 1040 format(6(e10.4,2x))

      return 
      end 

c===========================================================
c
      subroutine laplacian(u_xx,u,nx,dx,ipdmp,ier)

c     Arguments

      integer 
     &     nx,               ! dimension of the array u
     &     ipdmp,ier         ! dump unit, error flag

      real
     &     dx,               ! x step
     &     u(nx),            ! input vector 
     &     u_xx(nx)           ! output vector (xx derivative)

c     Internal Variables

      integer 
     &     i                 ! index of array 

      real 
     &     dx2               ! square of dx 

c--------------------------------------------------------

      dx2 = dx*dx

      do i=2,nx-1
         u_xx(i)=(u(i+1)-2*u(i)+u(i-1))/dx2
      end do
      u_xx(1)=(6*u(4)-23*u(3)+28*u(2)-u(1))/(-5*dx2)
      u_xx(nx)=(6*u(4)-23*u(3)+28*u(2)-u(1))/(5*dx2)

      return 
      end 

c========================================================
      subroutine der_cen_z(u_z,u,nx,nz,dz,ipdmp,ier)

c     Arguments

      integer 
     &     nx,nz,                ! dimension of the array u
     &     ipdmp,ier             ! dump unit, error flag

      real
     &     dz,                   ! z steps
     &     uzmin,uzmax,u(nz,nx), ! min, max of uz, input field
     &     u_z(nz,nx)            ! output field (z derivative)

c     Internal Variables

      integer 
     &     i,j                  ! index of array 

c--------------------------------------------------------

      do i=1,nx
         do j=2,nz-1
            u_z(j,i)=(u(j+1,i) - u(j-1,i))/(2*dz)
         end do  
         u_z(nz,i)=(u(nz-2,i)-4*u(nz-1,i)+3*u(nz,i))/(2*dz)
         u_z(1,i)=(u(3,i)-4*u(2,i)+3*u(1,i))/(-2*dz)
      end do

      uzmax=u_z(1,1)
      uzmin=u_z(1,1)

      return 
      end
c***********************************************************
      subroutine interp_lin(b,a1,a2,nx,alpha,ipdmp,ier)
c===========================================================
c
c     This subroutine interpolates linearly between
c     the vectors a1(nx) and a2(nx). The result is put in 
c     the vector b(alpha+1,nx)
c===========================================================
      
c---
c Arguments 
c---
      integer 
     &     nx,                  ! dimension of the array a1,a2
     &     alpha,               ! step variable
     &     ipdmp,ier            ! dump unit, error flag

      real
     &     a1(nx),              ! first vector
     &     a2(nx),              ! second vector
     &     b((alpha+1)*nx)        ! output array

c---
c Local Variables
c---      

      real
     &     beta,                ! inverse of alpha
     &     dist                 ! junk variable

      integer
     &     i,j            ! index
      
c----------------------------------------------------------

      beta=1./alpha

      do i=1,nx
         do j=1,alpha
            dist=(j-1)*beta
            b(i+(j-1)*nx)=(1-dist)*a1(i)+dist*a2(i)
         enddo
         b(i+alpha*nx)=a2(i)
      enddo

 1040 format(6(e10.4,2x))

      return
      end

c===========================================================
